/*********************************************************************************
 * Copyright (c) 2006 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 
package de.fzj.unicore.bes.functional;

import java.io.File;
import java.io.IOException;

import org.apache.xmlbeans.XmlException;
import org.codehaus.xfire.fault.XFireFault;
import org.ggf.schemas.bes.x2006.x08.besFactory.ActivityDocumentType;
import org.ggf.schemas.bes.x2006.x08.besFactory.ActivityStateEnumeration;
import org.ggf.schemas.bes.x2006.x08.besFactory.ActivityStatusType;
import org.ggf.schemas.bes.x2006.x08.besFactory.CreateActivityDocument;
import org.ggf.schemas.bes.x2006.x08.besFactory.CreateActivityResponseDocument;
import org.ggf.schemas.bes.x2006.x08.besFactory.GetActivityStatusesDocument;
import org.ggf.schemas.bes.x2006.x08.besFactory.GetActivityStatusesResponseDocument;
import org.ggf.schemas.bes.x2006.x08.besFactory.GetActivityStatusesResponseType;
import org.ggf.schemas.bes.x2006.x08.besFactory.GetActivityStatusesType;
import org.ggf.schemas.bes.x2006.x08.besManagement.StartAcceptingNewActivitiesDocument;
import org.ggf.schemas.bes.x2006.x08.besManagement.StopAcceptingNewActivitiesDocument;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDefinitionDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.bes.BESFactory;
import de.fzj.unicore.bes.BESManagement;
import de.fzj.unicore.bes.client.FactoryClient;
import de.fzj.unicore.bes.faults.InvalidRequestMessageFault;
import de.fzj.unicore.bes.faults.NotAcceptingNewActivitiesFault;
import de.fzj.unicore.bes.faults.UnknownActivityIdentifierFault;
import de.fzj.unicore.bes.faults.UnsupportedFeatureFault;
import de.fzj.unicore.bes.security.SecurityConstant;
import de.fzj.unicore.uas.client.BaseUASClient;
import de.fzj.unicore.wsrflite.xfire.WSRFClientFactory;
import eu.unicore.util.httpclient.IClientConfiguration;


/**
 * Test class to verify BES interfaces with username/password tokens.
 * 
 * @author m.memon
 */
public class TestUserNameActivity extends BaseActivityTest {

	private static final String USERNAME = "ogf";
	private static String PASSWORD = "ogf";
	private IClientConfiguration oldSecProp;
	FactoryClient factory;

	@Override
	protected void setUp() throws Exception {
		super.setUp();
		setSecurityFlag(true);
		configureSecurity();
		oldSecProp = sp.clone(); 
		sp.getExtraSettings().setProperty(SecurityConstant.ATTR_USERNAME, USERNAME);
		sp.getExtraSettings().setProperty(SecurityConstant.ATTR_PASSWORD, PASSWORD);
		sp.getExtraSettings().setProperty(SecurityConstant.ATTR_ACTIVATEUSERNAMEPROFILE, "true");
		if (factory == null)
			factory = getDefaultFactory();
	}

	
	private EndpointReferenceType getUVaFactoryEpr(){
		try {
			EndpointReferenceType eprt = EndpointReferenceType.Factory.parse(new File("/tmp/uva/sulla.xml"));
			return eprt;
		} catch (XmlException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return null;
	}
	
	private BESFactory getUserNameFactoryUVa(String u, String p) {
		String url = "https://128.143.136.138:18443/axis/services/GeniiBESPortType?genii-container-id=51966AB2-A4EB-FC04-DC02-B7B9F2CC8937";
		try {
			configureSecurity();
			BaseUASClient besClient = new BaseUASClient(url, getUVaFactoryEpr(),sp); 
			BESFactory factory = null;
			factory = besClient.makeProxy(BESFactory.class);
			return factory;
		} catch (Exception e) {
			e.printStackTrace();
			fail(e.toString());
		}
		return null;
	}


	/**
	 * This method test activity creation with <b>Username/Password</b> 
	 * credentials transported through <code>UsernameOutHandler</code> and validted through 
	 * <code>UsernameOutHandler</code>.
	 * @throws IOException 
	 * @throws XmlException 
	 * @throws CloneNotSupportedException 
	 * 
	 */
	public void testCreateActivity() throws XmlException, IOException, CloneNotSupportedException {

		CreateActivityDocument activityDoc = CreateActivityDocument.Factory.newInstance();
		ActivityDocumentType activityType = activityDoc.addNewCreateActivity().addNewActivityDocument();
		activityType.addNewJobDefinition();
		
		JobDefinitionDocument jobDef = JobDefinitionDocument.Factory.parse(new File("/home/m.memon/besws/ogsabes-core/src/test/resources/jsdl/sleep-hpcp.xml"));
//		JobDefinitionDocument jobDef = JobDefinitionDocument.Factory.parse(new File("/tmp/ogf27-jsdl/*"));
		
		activityDoc.getCreateActivity().getActivityDocument().setJobDefinition(jobDef.getJobDefinition());
		
		CreateActivityResponseDocument res = null;
		
		try {
			res = factory.createActivity(activityDoc);
			assertFalse(res.getCreateActivityResponse().getActivityIdentifier()==null);
		} catch (NotAcceptingNewActivitiesFault e) {
			e.printStackTrace();
		} catch (InvalidRequestMessageFault e) {
			e.printStackTrace();
		} catch (UnsupportedFeatureFault e) {
			e.printStackTrace();
		}catch(Exception e){
			e.printStackTrace();
			fail();
		}
		
		
		int seed = 2000;
		long maxTimeLimit = 10*60*1000; // max allowed limit 10 minutes
		long accTimeLimit = 0L;
		
		while (true){
			ActivityStatusType status = getActivityStatus(res.getCreateActivityResponse().getActivityIdentifier()).getResponseArray()[0].getActivityStatus();
			
			if(status.getState() == ActivityStateEnumeration.FINISHED){
				assertTrue("Activity Finished Successfully.", true);
				break;
			}
			
			else if (status.getState() == ActivityStateEnumeration.FAILED) {
				// better to test execution in this test 
				assertTrue("Activity Failed.", false);
				break;
			}
				
			else {
				try {
					accTimeLimit += seed;
					if(accTimeLimit >= maxTimeLimit){
						assertTrue("Activity execution time exceeded the allotted total time.", false);
						break;
					}
					Thread.sleep(seed);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
			
		}
		
		
		BaseUASClient b = null;
		try {
			b = new BaseUASClient(serverUrl+"BESFactory", getDefaultFactoryEpr(), oldSecProp);
			b.getResourcePropertyDocument();
			fail();
		} catch (Exception e1) {
			// on success, code must reach here 
			assertTrue(true);
		}

	}
	
	
	protected GetActivityStatusesResponseType getActivityStatus(EndpointReferenceType eprt){
		GetActivityStatusesDocument getActStatReq = GetActivityStatusesDocument.Factory.newInstance();
		GetActivityStatusesType getActStatReqType =  getActStatReq.addNewGetActivityStatuses();
		getActStatReqType.setActivityIdentifierArray(new EndpointReferenceType[]{eprt});
		GetActivityStatusesResponseDocument getActStatRes;
		try {
			 getActStatRes = factory.getActivityStatuses(getActStatReq);
			return getActStatRes.getGetActivityStatusesResponse();
		} catch (UnknownActivityIdentifierFault e) {
			e.printStackTrace();
			fail("Couldn't activity status: "+e.getMessage());
		}
		return null;
	}
	
	
	/**
	 * This method test BESManagement functions with <b>Username/Password</b> tokens. To see how {@link BESFactory} behaves while jobs being controlled by {@link BESManagement} 
	 * then please check {@link TestBESManagement}  check how job creation behaves 
	 */
	public void testManagedFactory(){
		
		WSRFClientFactory mFactory = new WSRFClientFactory(sp);
		
		BESManagement mObj = null;
		
		String rprop;
			
		try {
			rprop = factory.getResourceProperty(BESFactory.RPIsAcceptingNewActivities);
			mObj = mFactory.createPlainWSProxy(BESManagement.class, serverUrl+"BESManagement");
		} catch (Exception e) {
			e.printStackTrace();
			fail();
		}

		StopAcceptingNewActivitiesDocument stopDoc = StopAcceptingNewActivitiesDocument.Factory.newInstance();
		try {
			stopDoc.addNewStopAcceptingNewActivities();
			mObj.StopAcceptingNewActivities(stopDoc);
		} catch (XFireFault e1) {
			e1.printStackTrace();
			fail();
		}

		try {
			rprop = factory.getResourceProperty(BESFactory.RPIsAcceptingNewActivities);
			assertTrue(rprop.contains("false"));
		} catch (Exception e2) {
			e2.printStackTrace();
			fail();
		}

		StartAcceptingNewActivitiesDocument startDoc =
		StartAcceptingNewActivitiesDocument.Factory.newInstance();
		startDoc.addNewStartAcceptingNewActivities();
		 try {
			mObj.StartAcceptingNewActivities(startDoc);
		} catch (XFireFault e) {
			e.printStackTrace();
		}

		try {
			rprop = factory.getResourceProperty(BESFactory.RPIsAcceptingNewActivities);
			assertTrue(rprop.contains("true"));
		} catch (Exception e2) {
			e2.printStackTrace();
			fail();
		}
		
	}
}


