package de.fzj.unicore.bes.activity.rp;

import javax.xml.namespace.QName;

import org.ggf.schemas.bes.x2006.x08.besFactory.ActivityStateEnumeration;
import org.ggf.schemas.bes.x2006.x08.besFactory.ActivityStatusDocument;
import org.ggf.schemas.bes.x2006.x08.besFactory.ActivityStatusType;
import org.ogf.schemas.hpcp.x2007.x01.fs.ExecutingDocument;
import org.ogf.schemas.hpcp.x2007.x01.fs.QueuedDocument;
import org.ogf.schemas.hpcp.x2007.x01.fs.ReadyDocument;
import org.ogf.schemas.hpcp.x2007.x01.fs.StagingInDocument;
import org.ogf.schemas.hpcp.x2007.x01.fs.StagingOutDocument;
import org.ogf.schemas.hpcp.x2007.x01.fs.StartingDocument;
import org.xmlsoap.schemas.soap.envelope.Fault;

import de.fzj.unicore.bes.BESFactory;
import de.fzj.unicore.bes.util.BESUtils;
import de.fzj.unicore.uas.xnjs.XNJSFacade;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import de.fzj.unicore.xnjs.ems.ActionResult;
import de.fzj.unicore.xnjs.ems.ActionStatus;

/**
 * 
 * This class converts XNJS ActionState to the OGSA-BES activity state model.
 * As BES's state model defines only the basic states (see <a href="http://www.ogf.org/documents/GFD.108.pdf">GFD.108</a>), 
 * to make it usable for clients we extended it by introducing useful sub-states.     
 * 
 * <table>
 * <col width="30%"/>
 * <col width="70%"/>
 * <tbody>
 * 	  <tr><td><b>XNJS State</b></td><td><b>OGSA-BES State</b></th></tr>
 *    <tr><td>CREATED</td><td>Pending</td></tr>
 *    <tr><td>QUEUED</td><td>Running:Queued</td></tr>
 *    <tr><td>PREPROCESSING</td><td>Running:Staging-In</td></tr>
 *    <tr><td>READY</td><td>Running:Ready</td></tr>
 *    <tr><td>PENDING</td><td>Running:Starting</td></tr>
 *    <tr><td>RUNNING</td><td>Running:Executing</td></tr>
 *    <tr><td>POSTPROCESSING</td><td>Running:Staging-out</td></tr>
 *    <tr><td>DONE (If successful)</td><td>Finished</td></tr>
 *    <tr><td>DONE (If user aborted)</td><td>Canceled</td></tr>
 *    <tr><td>DONE (If failed)</td><td>Failed (returned with a soap fault element)</td></tr>
 * </tbody>
 * </table>
 * 
 * */

public final class BESActivityConverter {

	/**
	 * 
	 * This class converts XNJS ActionState to the BES activity state model.
	 * 
	 * @param xnjsReference
	 * @param actionId
	 * @param status
	 * @return activity status complaint with OGSA-BES 
	 */
	public synchronized static ActivityStatusDocument getBESStatefromXNJSState(Kernel k, String xnjsReference, String actionId, Integer status) {
		ActivityStatusType aType = ActivityStatusType.Factory.newInstance();
		ActivityStatusDocument aDoc = ActivityStatusDocument.Factory.newInstance();
		
		switch (status) {
		
		// CREATED -> Pending
		case ActionStatus.CREATED:
			aType.setState(ActivityStateEnumeration.PENDING);
			aDoc.setActivityStatus(aType);
			break;

		// No mapping	
		case ActionStatus.FROZEN:
		case ActionStatus.PAUSED:
			break;

		// QUEUED -> Running:Queued 
		case ActionStatus.QUEUED:
			QueuedDocument qDoc = QueuedDocument.Factory.newInstance();
			qDoc.addNewQueued();
			aType.setState(ActivityStateEnumeration.RUNNING);
			aDoc.setActivityStatus(aType);
			WSServerUtilities.append(qDoc, aDoc);
			break;
			
			

		// PREPROCESSING -> Running:Staging-In
		case ActionStatus.PREPROCESSING:
			aType.setState(ActivityStateEnumeration.RUNNING);
			aDoc.setActivityStatus(aType);
			StagingInDocument staginDoc = StagingInDocument.Factory.newInstance();
			staginDoc.addNewStagingIn();
			WSServerUtilities.append(staginDoc, aDoc);
			break;

		// READY -> Running:Ready
		case ActionStatus.READY:
			aType.setState(ActivityStateEnumeration.RUNNING);
			aDoc.setActivityStatus(aType);
			ReadyDocument rDoc = ReadyDocument.Factory.newInstance();
			rDoc.addNewReady();
			WSServerUtilities.append(rDoc, aDoc);
			break;

			
		// PENDING -> Running:Starting
		case ActionStatus.PENDING:
			aType.setState(ActivityStateEnumeration.RUNNING);
			aDoc.setActivityStatus(aType);
			StartingDocument sDoc = StartingDocument.Factory.newInstance();
			sDoc.addNewStarting();
			WSServerUtilities.append(sDoc, aDoc);
			break;
		
		// RUNNING -> Running:Executing
		case ActionStatus.RUNNING:
			ExecutingDocument execDoc = ExecutingDocument.Factory.newInstance();
			execDoc.addNewExecuting();
			aType.setState(ActivityStateEnumeration.RUNNING);
			aDoc.setActivityStatus(aType);
			WSServerUtilities.append(execDoc, aDoc);
			break;
		// POSTPROCESSING -> Running:Staging-out	
		case ActionStatus.POSTPROCESSING:
			StagingOutDocument stagoutDoc = StagingOutDocument.Factory.newInstance();
			stagoutDoc.addNewStagingOut();
			aType.setState(ActivityStateEnumeration.RUNNING);
			aDoc.setActivityStatus(aType);
			WSServerUtilities.append(stagoutDoc, aDoc);
			break;
		// mapping DONE to Finished/Cancelled/Failed
		case ActionStatus.DONE:
			if (XNJSFacade.get(xnjsReference, k).getAction(actionId).getResult().isSuccessful()){
				aType.setState(ActivityStateEnumeration.FINISHED);
				int exitCode = XNJSFacade.get(xnjsReference, k).getAction(actionId).getResult().getExitCode();
				aType.setExitCode(exitCode);
				aDoc.setActivityStatus(aType);
			} 
			else if(XNJSFacade.get(xnjsReference, k).getAction(actionId).getResult().getStatusCode()==ActionResult.USER_ABORTED){
				aType.setState(ActivityStateEnumeration.CANCELLED);
				aDoc.setActivityStatus(aType);
			}
				
			else{
				aType.setState(ActivityStateEnumeration.FAILED);
				int exitCode = XNJSFacade.get(xnjsReference, k).getAction(actionId).getResult().getExitCode();
				String message = XNJSFacade.get(xnjsReference, k).getAction(actionId).getResult().getErrorMessage();
				Fault fault = BESUtils.buildSOAP11Fault(new QName(BESFactory.BES_NS, "UnknownActivityExecutionFault"),message);
				aType.setExitCode(exitCode);
				aType.setFault(fault);
				aDoc.setActivityStatus(aType);
			}
			break;

		
		}
		
		return aDoc;
	}
	
	
}
