package de.fzj.unicore.bes.client;

import java.math.BigInteger;
import java.util.Calendar;

import org.apache.log4j.Logger;
import org.ggf.schemas.bes.x2006.x08.besFactory.ActivityStateEnumeration;
import org.ggf.schemas.bes.x2006.x08.besFactory.ActivityStatusDocument;
import org.unigrids.services.atomic.types.StatusInfoDocument;
import org.unigrids.x2006.x04.services.jms.AbortDocument;
import org.unigrids.x2006.x04.services.jms.HoldDocument;
import org.unigrids.x2006.x04.services.jms.ResumeDocument;
import org.unigrids.x2006.x04.services.jms.StartDocument;
import org.unigrids.x2006.x04.services.jms.SubmissionTimeDocument;
import org.unigrids.x2006.x04.services.jms.WorkingDirectoryReferenceDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.bes.BESActivity;
import de.fzj.unicore.uas.JobManagement;
import de.fzj.unicore.uas.client.BaseUASClient;
import de.fzj.unicore.uas.client.StorageClient;
import eu.unicore.util.Log;
import eu.unicore.util.httpclient.IClientConfiguration;

public class ActivityClient extends BaseUASClient {

	private static final Logger logger=Log.getLogger(Log.CLIENT, ActivityClient.class);

	private final ResumeDocument resumeDoc;
	private final HoldDocument holdDoc;
	private final AbortDocument abortDoc;
	private final StartDocument startDoc;
	private Calendar submissionTime;

	private final BESActivity activity;


	public ActivityClient(String url, EndpointReferenceType epr, IClientConfiguration sec) 
			throws Exception {
		super(url, epr, sec);
		activity = makeProxy(BESActivity.class);
		holdDoc=HoldDocument.Factory.newInstance(); holdDoc.addNewHold();
		abortDoc=AbortDocument.Factory.newInstance(); abortDoc.addNewAbort();
		startDoc=StartDocument.Factory.newInstance(); startDoc.addNewStart();
		resumeDoc=ResumeDocument.Factory.newInstance(); resumeDoc.addNewResume();
	}

	/**
	 * start this job<br/> This will throw an exception if the job is not in
	 * the READY state
	 */
	public void start() throws Exception {
		logger.info("Calling service at wsaTo: "+ epr.getAddress().getStringValue());
		activity.Start(startDoc);
	}

	public void abort() throws Exception {
		logger.debug("Calling service at wsaTo: "
				+ epr.getAddress().getStringValue());
		activity.Abort(abortDoc);
	}

	public void hold() throws Exception {
		logger.debug("Calling service at wsaTo: "
				+ epr.getAddress().getStringValue());
		activity.Hold(holdDoc);
	}

	public void resume() throws Exception {
		logger.debug("Calling service at wsaTo: "
				+ epr.getAddress().getStringValue());
		activity.Resume(resumeDoc);
	}



	/**
	 * convenience method that waits until a job has finished and returns the
	 * final status (SUCCESSFUL or FAILED)
	 * 
	 * @param timeout
	 *            in milliseconds (null for no timeout)
	 * @return
	 */
	public String waitUntilDone(int timeout) throws Exception {
		ActivityStateEnumeration.Enum status = ActivityStateEnumeration.PENDING;
		long start = System.currentTimeMillis();
		long elapsed = 0;
		while (true) {
			if (timeout > 0 && elapsed > timeout)
				break;
			elapsed = System.currentTimeMillis() - start;
			status = getStatus();
			if (status == ActivityStateEnumeration.FINISHED || status == ActivityStateEnumeration.FAILED){
                                break;
			}
			Thread.sleep(500);
		}
		return status.toString();
	}

	/**
	 * convenience method that waits until a job is READY and can be started
	 * 
	 * @param timeout
	 *            in milliseconds (null for no timeout)
	 * @return
	 */
	public String waitUntilReady(int timeout) throws Exception {
		ActivityStateEnumeration.Enum status = ActivityStateEnumeration.PENDING;
		long start = System.currentTimeMillis();
		long elapsed = 0;
		while (true) {
			if (timeout > 0 && elapsed > timeout)
				break;
			elapsed = System.currentTimeMillis() - start;
			status = getStatus();
			if (status.equals(ActivityStateEnumeration.RUNNING))
				break;
			if (status.equals(ActivityStateEnumeration.FAILED)
					|| status.equals(ActivityStateEnumeration.FINISHED)) {
				throw new Exception("Job is already done, status is <"
						+ status.toString() + ">");
			}
			Thread.sleep(500);
		}
		return status.toString();
	}

	/**
	 * return the job status
	 */
	public ActivityStateEnumeration.Enum getStatus()throws Exception{
		return ActivityStatusDocument.Factory.parse(getResourceProperty(BESActivity.RPStatus)).getActivityStatus().getState();
	}


	/**
	 * return the exit code of the job
	 * 
	 * @return the exit code or null if not available
	 * @throws Exception
	 */
	public Integer getExitCode()throws Exception{
		BigInteger exitCode = StatusInfoDocument.Factory.parse(
				getResourceProperty(JobManagement.RPStatusInfo))
				.getStatusInfo().getExitCode();
		return exitCode != null ? exitCode.intValue() : null;
	}

	/**
	 * return the submission time of the job
	 * @return
	 * @throws Exception
	 */
	public Calendar getSubmissionTime()throws Exception{
		if (submissionTime == null) {
			submissionTime = SubmissionTimeDocument.Factory.parse(
					getResourceProperty(JobManagement.RPSubmissionTime))
					.getSubmissionTime();

		}
		return submissionTime;
	}

	/**
	 * get a client for the Uspace SMS
	 * 
	 * @return StorageClient
	 * @throws Exception
	 */
	public StorageClient getUspaceClient() throws Exception {
		String address = getResourceProperty(JobManagement.RPWorkingDir);
		WorkingDirectoryReferenceDocument uEpr = WorkingDirectoryReferenceDocument.Factory
		.parse(address);
		StorageClient storageClient = new StorageClient(uEpr.getWorkingDirectoryReference(),
				getSecurityConfiguration());
		return storageClient;
	}

}
