package de.fzj.unicore.cisprovider.sensors;

import java.math.BigInteger;
import java.util.concurrent.TimeUnit;

import de.fzj.unicore.cisprovider.MonitoringEvent;
import de.fzj.unicore.cisprovider.Topics;
import de.fzj.unicore.cisprovider.impl.DefaultMonitoringEvent;
import de.fzj.unicore.cisprovider.impl.DefaultSensor;
import de.fzj.unicore.cisprovider.xmlbeans.JobsDocument;
import de.fzj.unicore.cisprovider.xmlbeans.JobsDocument.Jobs;
import de.fzj.unicore.uas.xnjs.XNJSFacade;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.xnjs.tsi.IExecutionSystemInformation;

/**
 * overall jobs running on the system (or rather a single XNJS)
 * 
 * @author schuller
 */
public class JobInfo extends DefaultSensor {
	
	private volatile int totalJobs;
	private volatile int runningJobs;
	private volatile int waitingJobs;
	
	private boolean dirty;
	
	public static int DEFAULT_NOTIFICATION=10;
	
	@Override
	public String getName() {
		return "Jobs";
	}

	@Override
	protected void setLastEvent(MonitoringEvent e) {
		dirty=true;
		super.setLastEvent(e);
	}

	@Override
	protected void doNotify() {
		if(dirty){
			dirty=false;
			super.doNotify();
		}
	}

	/**
	 * create a new JobInfo sensor that uses the default notification
	 * interval
	 */
	public JobInfo(Kernel kernel){
		this(kernel, DEFAULT_NOTIFICATION);
		
	}

	/**
	 * Create a jobinfo sensor that notifies its listeners at the specified interval
	 * 
	 * @param interval -  the minimum time between subsequent notifications
	 */
	public JobInfo(Kernel kernel, int interval){
		super(kernel);
		setLastEvent(createEvent());
		//periodically check job numbers, otherwise might miss events such as destroy
		Runnable command=new Runnable(){
			public void run(){
				update();
			}
		};
		kernel.getContainerProperties().getThreadingServices().getScheduledExecutorService().
			scheduleWithFixedDelay(command, 10, 10, TimeUnit.SECONDS);
	}

	private void update(){
		IExecutionSystemInformation sysInfo=XNJSFacade.get(null,kernel).getConfiguration().getComponentInstanceOfType(IExecutionSystemInformation.class);
		//get total number of jobs on the system, as seen by UNICORE 
		totalJobs=sysInfo.getTotalNumberOfJobs();
		runningJobs=sysInfo.getNumberOfRunningJobs();
		waitingJobs=sysInfo.getNumberOfQueuedJobs();
		setLastEvent(createEvent());
		doNotify();
	}
	
	protected DefaultMonitoringEvent createEvent(){
		DefaultMonitoringEvent e=new DefaultMonitoringEvent();
		e.setDescription("This contains information about the jobs on the site");
		e.setTopic(Topics.JOBS+"/General");
		e.setTitle("Job information");
		e.setDetails(makeDetailedJobInfo());
		if(getLastEvent()!=null)	{
			e.setSupersedes(getLastEvent().getID());
		}
		return e;
	}
	
	protected JobsDocument makeDetailedJobInfo(){
		JobsDocument jd=JobsDocument.Factory.newInstance();
		Jobs j=jd.addNewJobs();
		j.setTotalJobs(BigInteger.valueOf(totalJobs));
		j.setRunningJobs(BigInteger.valueOf(runningJobs));
		j.setWaitingJobs(BigInteger.valueOf(waitingJobs));
		return jd;
	}

}
