/*********************************************************************************
 * Copyright (c) 2007 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/

package de.fzj.unicore.cisprovider.impl;

import java.io.FileOutputStream;
import java.util.concurrent.TimeUnit;

import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.cisprovider.CIPProperties;
import de.fzj.unicore.cisprovider.CISInfoProvider;
import de.fzj.unicore.cisprovider.Monitor;
import de.fzj.unicore.cisprovider.sensors.ApplicationsInfo;
import de.fzj.unicore.cisprovider.sensors.ExecutionEnvironmentInfo;
import de.fzj.unicore.cisprovider.sensors.JobInfo;
import de.fzj.unicore.cisprovider.sensors.SiteInfo;
import de.fzj.unicore.cisprovider.sensors.TextInfo;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import de.fzj.unicore.wsrflite.xfire.XFireKernel;
import de.fzj.unicore.wsrflite.xmlbeans.registry.RegistryHandler;

/**
 * setup CIP, optionally publish CIP web service to registry
 */
public class InitOnStartup implements Runnable {

	private final Kernel kernel;
	
	public InitOnStartup(Kernel kernel){
		this.kernel=kernel;
	}
	
	public void run(){
		CIPProperties cipCfg = new CIPProperties(kernel.getContainerProperties().getRawProperties());
		kernel.addConfigurationHandler(CIPProperties.class, cipCfg);

		Monitor m=cipCfg.getMonitor();
		m.addSensor(new SiteInfo(kernel));
		m.addSensor(new JobInfo(kernel));
		m.addSensor(new ApplicationsInfo(kernel));
		m.addSensor(new ExecutionEnvironmentInfo(kernel));
		m.addSensor(new TextInfo(kernel));
		
		initGlue2Generator();

		if(!cipAvailable()){
			CISInfoProviderImpl.logger.warn("CIS Infoprovider is not deployed, " +
			"not publishing service info to the registry");
			return;
		}
		
		publish();
	}

	protected boolean cipAvailable(){
		return XFireKernel.get(kernel).getService(CISInfoProvider.SERVICENAME)!=null;
	}
	
	/**
	 * publishes CIP web service in the registry <br/>
	 */
	protected void publish() {
		try {
			//check if publishing is disabled
			CIPProperties cfg = kernel.getAttribute(CIPProperties.class);
			if(!cfg.getBooleanValue(CIPProperties.PUBLISH)){
				CISInfoProviderImpl.logger.info("NOT publishing CIP to registry.");
				return;
			}
			EndpointReferenceType epr = WSServerUtilities.newEPR(kernel.getContainerSecurityConfiguration());
			epr.addNewAddress().setStringValue(WSServerUtilities.makeAddress(CISInfoProvider.SERVICENAME, 
					kernel.getContainerProperties()));
			WSServerUtilities.addPortType(epr, CISInfoProvider.PORTTYPE);
			RegistryHandler rh = kernel.getAttribute(RegistryHandler.class);
			rh.getRegistryClient().addRegistryEntry(epr);
			CISInfoProviderImpl.logger.info("Added registry entry for " + CISInfoProvider.SERVICENAME);

		} catch (Exception e) {
			LogUtil.logException("Could not publish to registry", e, CISInfoProviderImpl.logger);
		}
	}

	/**
	 * if enabled, periodically generate GLUE2 doc and store it at specified location
	 */
	protected void initGlue2Generator(){
		CIPProperties cipCfg = kernel.getAttribute(CIPProperties.class);
		boolean enabled=cipCfg.getBooleanValue(CIPProperties.GLUE2_GENERATE);
		if(enabled){
			String path=cipCfg.getValue(CIPProperties.GLUE2_TARGET);
			long period=cipCfg.getLongValue(CIPProperties.GLUE2_REFRESHPERIOD);
			Runnable command=new Glue2Generate(kernel);
			kernel.getContainerProperties().getThreadingServices().
				getScheduledExecutorService().scheduleWithFixedDelay(command, 5, period, TimeUnit.SECONDS);
			
			CISInfoProviderImpl.logger.info("Generating GLUE2 service description at <"+path+"> every "+period+" seconds.");
		}
	}
	
	public static class Glue2Generate implements Runnable{
		
		final Kernel kernel;
		
		public Glue2Generate(Kernel kernel){
			this.kernel=kernel;
		}
		
		public void run(){
			try{
				CIPProperties cfg = kernel.getAttribute(CIPProperties.class);
				String path=cfg.getValue(CIPProperties.GLUE2_TARGET);
				FileOutputStream fos=new FileOutputStream(path);
				String unique=kernel.getContainerProperties().getBaseUrl();
				try{
					(new GlueInstanceGenerator().generate(cfg,unique)).save(fos);
				}finally{
					fos.close();
				}
			}catch(Exception ex){
				//need to stop the publisher?!
			}
		}
	}
	
}
