package de.fzj.unicore.cisprovider.impl;

import java.io.File;
import java.io.IOException;
import java.math.BigInteger;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlAnySimpleType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ApplicationType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ResourcesDocument;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ResourcesType;
import org.json.JSONException;
import org.json.JSONObject;
import org.ogf.schemas.glue.x2009.x03.spec20R1.AdminDomainT;
import org.ogf.schemas.glue.x2009.x03.spec20R1.ApplicationEnvironmentT;
import org.ogf.schemas.glue.x2009.x03.spec20R1.ApplicationEnvironmentsDocument.ApplicationEnvironments;
import org.ogf.schemas.glue.x2009.x03.spec20R1.ComputingEndpointT;
import org.ogf.schemas.glue.x2009.x03.spec20R1.ComputingManagerT;
import org.ogf.schemas.glue.x2009.x03.spec20R1.ComputingServiceT;
import org.ogf.schemas.glue.x2009.x03.spec20R1.DomainsDocument;
import org.ogf.schemas.glue.x2009.x03.spec20R1.EndpointHealthStateT;
import org.ogf.schemas.glue.x2009.x03.spec20R1.ExecutionEnvironmentT;
import org.ogf.schemas.glue.x2009.x03.spec20R1.ExtendedBooleanT;
import org.ogf.schemas.glue.x2009.x03.spec20R1.ExtensionT;
import org.ogf.schemas.glue.x2009.x03.spec20R1.ExtensionsT;
import org.ogf.schemas.glue.x2009.x03.spec20R1.LocationT;
import org.ogf.schemas.glue.x2009.x03.spec20R1.QualityLevelT;
import org.ogf.schemas.glue.x2009.x03.spec20R1.ServingStateT;
import org.ogf.schemas.glue.x2009.x03.spec20R1.StorageServiceCapacityT;
import org.ogf.schemas.glue.x2009.x03.spec20R1.StorageServiceT;
import org.unigrids.services.atomic.types.TextInfoType;

import de.fzj.unicore.cisprovider.CIPConstants;
import de.fzj.unicore.cisprovider.CIPProperties;
import de.fzj.unicore.cisprovider.Monitor;
import de.fzj.unicore.cisprovider.MonitoringEvent;
import de.fzj.unicore.cisprovider.Topics;
import de.fzj.unicore.cisprovider.util.DateUtil;
import de.fzj.unicore.cisprovider.xmlbeans.JobsDocument;
import de.fzj.unicore.cisprovider.xmlbeans.SiteDocument;
import de.fzj.unicore.uas.util.LogUtil;

public class GlueInstanceGenerator {

	private static final Logger logger = LogUtil.getLogger(CIPConstants.LOG_PREFIX, GlueInstanceGenerator.class);

	private DomainsDocument gDoc = null;
	
	private JSONObject jsObj = null;

	private String emiVersionFile="/etc/emi-version";
	
	public static final String SERVICE_TYPE="eu.unicore.USE";
	
	public GlueInstanceGenerator(){}
	
	/**
	 * generate GLUE2 doc
	 * @param cfg - CIP configuration
	 */
	public DomainsDocument generate(CIPProperties cfg, String unique) {
		Monitor m = cfg.getMonitor();
		String cipDataPath=cfg.getValue(CIPProperties.DATA_PATH);
		//TODO use more specific list of events, or use sensors directly
		List<MonitoringEvent> events = m.getEvents("/", null, null, 1000);
		return generate(events,cipDataPath,unique);
	}
	
	/**
	 * generate GLUE2 doc
	 * @param events - full list of events
	 * @param cipDataPath - path of JSON file containing basic info
	 */
	public DomainsDocument generate(List<MonitoringEvent> events, String cipDataPath, String unique) {
		String jsonString = null;
		try {
			if (cipDataPath != null && cipDataPath != "") {
				jsonString = readFileAsString(cipDataPath);
				jsObj = new JSONObject(jsonString);

			} else {
				logger.debug("cip is null, creating default json info");
				jsObj = createDefaultJSON();

			}
			jsObj = jsObj.getJSONObject("unicore-site");
		} catch (IOException e1) {
			logger.warn("Cannot read CIP data file", e1);
		} catch (JSONException e) {
			logger.warn("invalid JSON file", e);
		}

		gDoc = DomainsDocument.Factory.newInstance();
		AdminDomainT adminT = gDoc.addNewDomains().addNewAdminDomain();

		adminT.setBaseType(getBaseType("Domain"));
		if (adminT.getID() == null) {
			adminT.setID("urn:unicore:ad:"+unique);
		}

		adminT.addNewComputingService().addNewComputingManager();

		ComputingServiceT csType = null;
		ComputingManagerT cmType = null;
		try {
			csType = gDoc.getDomains().getAdminDomainArray(0)
					.getComputingServiceArray(0);
			cmType = csType.getComputingManagerArray(0);
		} catch (Exception e) {
			csType = gDoc.addNewDomains().addNewAdminDomain()
					.addNewComputingService();
			cmType = csType.addNewComputingManager();
		}

		if (csType.getID() == null) {
			csType.setID("urn:unicore:cs:"+unique);
		}

		csType.setBaseType(getBaseType("Service"));

		if (csType.getType() == null) {
			csType.setType(SERVICE_TYPE);
		}
		if (csType.getQualityLevel() == null) {
			csType.setQualityLevel(QualityLevelT.PRODUCTION);
		}

		if (cmType.getID() == null) {
			cmType.setID("urn:unicore:cm:"+unique);
		}

		cmType.setBaseType(getBaseType("Manager"));
		if (cmType.getProductName() == null)
			cmType.setProductName("unicore/x");

		for (Iterator<MonitoringEvent> iterator = events.iterator(); iterator
				.hasNext();) {
			MonitoringEvent monitoringEvent = iterator.next();
			setEndpointInfo(monitoringEvent, unique);
			setExecutionEnvironment(monitoringEvent);
			setApplications(monitoringEvent);
			setJobs(monitoringEvent);
			setExtensions(monitoringEvent);
		}
		
		setLocationInformation(unique);
		setStorageInformation(unique);
		setDowntimeInformation();
		
		return gDoc;
	}

	/**
	 * The method will return the default json string if the cip json file path
	 * is unspecified
	 * 
	 * @param JSONObject
	 * */
	private JSONObject createDefaultJSON() {
		JSONObject jsonObject = null;
		try {
			jsonObject = new JSONObject(
					"{\"unicore-site\":" +
					"{\"location\":" +
					"{\"longitude\":55," +
					"\"latitude\":2," +
					"\"city\":\"city_name\"," +
					"\"country\":\"country\"," +
					"\"name\":\"fzj-demo-site\"}," +
					"\"storage\":{\"total-capacity\":1024}," +
					"\"compute\":{\"platform\":\"x86\"}}}");
		} catch (JSONException e) {
			logger.error("", e);
		}
		return jsonObject;
	}

	private void setDowntimeInformation() {
		ComputingEndpointT ceT = null;
		JSONObject jsDowntime = null;
		try {
			ceT = gDoc.getDomains().getAdminDomainArray(0)
					.getComputingServiceArray(0).getComputingEndpointArray(0);
		} catch (IndexOutOfBoundsException e) {
			ceT = gDoc.getDomains().getAdminDomainArray(0)
					.getComputingServiceArray(0).addNewComputingEndpoint();
		}
		try {
			jsDowntime = jsObj.getJSONObject("downtime");
			String start = null;
			String end = null;
			String announce = null;
			String description = null;
			try {
				if ((start = jsDowntime.getString("start")) != null) {
					ceT.setDowntimeStart(DateUtil.toCalendar(start));
				}
			} catch (JSONException e) {
				// do nothing
			}
			try {
				if ((end = jsDowntime.getString("end")) != null) {
					ceT.setDowntimeEnd(DateUtil.toCalendar(end));
				}
			} catch (JSONException e) {

			}
			try {
				if ((announce = jsDowntime.getString("announce")) != null) {
					ceT.setDowntimeAnnounce(DateUtil.toCalendar(announce));
				}
			} catch (JSONException e) {
			}
			try {
				if ((description = jsDowntime.getString("description")) != null) {
					if (ceT.isSetDowntimeInfo()) {
						ceT.setDowntimeInfo(description);
					}

				}
			} catch (JSONException e) {
			}

		} catch (JSONException e) {
			// do nothing as it is an optional attribute
		}

	}

	private void setStorageInformation(String unique) {
		StorageServiceT ssT = null;
		StorageServiceCapacityT scT = null;
		try {
			ssT = gDoc.getDomains().getAdminDomainArray(0)
					.getStorageServiceArray(0);
		} catch (IndexOutOfBoundsException e) {
			if (ssT == null)
				ssT = gDoc.getDomains().getAdminDomainArray(0)
						.addNewStorageService();
		}
		ssT.setID("urn:unicore:storageservice:"+unique);
		ssT.setBaseType(getBaseType("Service"));
		ssT.setType(SERVICE_TYPE);
		ssT.setQualityLevel(QualityLevelT.PRODUCTION);

		JSONObject jsStorageObj;
		try {
			scT = ssT.getStorageServiceCapacityArray(0);
		} catch (IndexOutOfBoundsException e) {
			scT = ssT.addNewStorageServiceCapacity();
		}
		scT.setID("urn:unicore.sc1");
		scT.setType("online");
		try {
			jsStorageObj = jsObj.getJSONObject("storage");
			if (jsStorageObj != null) {
				String capacity = jsStorageObj.getString("total-capacity");
				if (capacity != null) {
					BigInteger bi = new BigInteger(capacity);
					scT.setTotalSize(bi);
				}
			}

		} catch (JSONException e) {
		}

	}

	private void setExtensions(MonitoringEvent monitoringEvent) {
		ExtensionsT extsT = null;
		if (gDoc.getDomains().getAdminDomainArray(0).getExtensions() == null) {
			extsT = gDoc.getDomains().getAdminDomainArray(0).addNewExtensions();

			// setting the extensions from simpleidb file

			if (monitoringEvent.getTopic().contains(Topics.TEXT + "/General")) {
				TextInfoType[] txtInfoTypeArr = (TextInfoType[]) monitoringEvent
						.getDetailsArray();
				if (txtInfoTypeArr != null) {

					extsT = gDoc.getDomains().getAdminDomainArray(0)
							.getExtensions();

					for (int i = 0; i < txtInfoTypeArr.length; i++) {
						ExtensionT extT = extsT.addNewExtension();
						extT.setLocalID("one-more-random-glue2-field");
						extT.setKey(txtInfoTypeArr[i].getName());
						extT.setValue(txtInfoTypeArr[i].getValue());
					}
				}

			}

			// setting extensions from cip json
			JSONObject jsExts = null;
			try {
				if ((jsExts = jsObj.getJSONObject("extensions")) != null) {

					for (Iterator<?> iterator = jsExts.sortedKeys(); iterator
							.hasNext();) {
						String key = (String) iterator.next();

						ExtensionT extT = extsT.addNewExtension();
						extT.setLocalID("glue2-is-da-bomb");
						extT.setKey(key);
						extT.setValue(jsExts.getString(key));
					}

				}
			} catch (JSONException e) {
				// do nothing
			}
		}
	}

	private void setJobs(MonitoringEvent monitoringEvent) {
		if (monitoringEvent.getTopic().contains(Topics.JOBS + "/General")) {
			// setting job info
			JobsDocument jd = (JobsDocument) monitoringEvent.getDetails();
			ComputingServiceT csType = gDoc.getDomains().getAdminDomainArray(0)
					.getComputingServiceArray(0);
			csType.setTotalJobs(jd.getJobs().getTotalJobs().longValue());
			csType.setRunningJobs(jd.getJobs().getRunningJobs().longValue());
		}
	}

	private  XmlAnySimpleType getBaseType(String typeName) {
		XmlAnySimpleType xasT = XmlAnySimpleType.Factory.newInstance();
		xasT.setStringValue(typeName);
		return xasT;
	}

	private void setExecutionEnvironment(MonitoringEvent monitoringEvent) {
		// setting resource info from simple-idb
		if (monitoringEvent.getTopic().contains(Topics.EXECUTION + "/General")) {
			ResourcesDocument rd = null;
			ResourcesType rt = null;
			try {
				rd = (ResourcesDocument) monitoringEvent.getDetails();
				rt = rd.getResources();
			} catch (Exception e) {
				logger.warn("error in getting resources' information from simple-idb");
				return;
			}

			ComputingManagerT cmType = gDoc.getDomains().getAdminDomainArray(0)
					.getComputingServiceArray(0).getComputingManagerArray(0);
			ExecutionEnvironmentT eEnvType = null;
			Double dPhysicalMemory = 0.0;
			try {
				if (cmType.isSetExecutionEnvironments()) {
					eEnvType = cmType.getExecutionEnvironments()
							.getExecutionEnvironmentArray(0);
				} else {
					eEnvType = cmType.addNewExecutionEnvironments()
							.addNewExecutionEnvironment();
				}

			} catch (IndexOutOfBoundsException e) {
				eEnvType = cmType.addNewExecutionEnvironments()
						.addNewExecutionEnvironment();
			}
			if (eEnvType.getPlatform() == null
					&& rt.getCPUArchitecture().getCPUArchitectureName() != null) {
				eEnvType.setPlatform(rt.getCPUArchitecture()
						.getCPUArchitectureName().toString());
			} else {
				eEnvType.setPlatform("x86");
			}

			if (!rt.isSetTotalPhysicalMemory()) {
				eEnvType.setMainMemorySize(BigInteger.valueOf(0));
			} else {
				dPhysicalMemory = rt.getTotalPhysicalMemory().getRangeArray(0)
						.getUpperBound().getDoubleValue();
				eEnvType.setMainMemorySize(BigInteger.valueOf(dPhysicalMemory
						.longValue()));

			}
			
			if (rt.isSetOperatingSystem()) {
				eEnvType.setOSFamily(rt.getOperatingSystem()
						.getOperatingSystemType().getOperatingSystemName() != null ? rt
						.getOperatingSystem().getOperatingSystemType()
						.getOperatingSystemName().toString()
						: "unknown");
				eEnvType.setOSName(rt.getOperatingSystem()
						.getOperatingSystemType().getOperatingSystemName() != null ? rt
						.getOperatingSystem().getOperatingSystemType()
						.getOperatingSystemName().toString()
						: "unknown");
				eEnvType.setOSVersion(rt.getOperatingSystem()
						.getOperatingSystemVersion() != null ? rt
						.getOperatingSystem().getOperatingSystemVersion() : "0");
			} else {
				//setting the mandatory attribute
				eEnvType.setOSFamily("unknown");
			}
			
			//TODO mandatory attributes
			eEnvType.setConnectivityIn(ExtendedBooleanT.FALSE);
			eEnvType.setConnectivityOut(ExtendedBooleanT.FALSE);
			
			eEnvType.setBaseType(getBaseType("Resource"));
			eEnvType.setID("urn:unicore.ee1");

			// setting physical total CPUs
			Long totalPhysicalCPU = getTotalCPUCount(rt);
			Long tmpPhysicalCPUs = eEnvType.getPhysicalCPUs();
			// The value defined in the site-info.glue document over-ride the
			// simpleidb
			// entry for the same attribute
			if (tmpPhysicalCPUs != null) {
				eEnvType.setPhysicalCPUs(totalPhysicalCPU);
			}

			try {
				Long clockSpeed = jsObj.getJSONObject("compute").getLong(
						"cpu-clock-speed");
				if (clockSpeed != null) {
					eEnvType.setCPUClockSpeed(jsObj.getJSONObject("compute")
							.getLong("cpu-clock-speed"));
				}

			} catch (JSONException e) {
				// do nothing
			}
		}
	}

	private Long getTotalCPUCount(ResourcesType rt) {
		Double dIndividualCPU = 0.0;
		Double dTotalResCount = 0.0;
		Long totalCPUCount = 0L;
		try {
			dIndividualCPU = rt.getIndividualCPUCount().getRangeArray(0)
					.getUpperBound().getDoubleValue();
			dTotalResCount = rt.getTotalResourceCount().getRangeArray(0)
					.getUpperBound().getDoubleValue();
			totalCPUCount = dIndividualCPU.longValue()
					* dTotalResCount.longValue();
		} catch (Exception e) {
			return 0L;
		}

		return totalCPUCount;
	}

	private void setApplications(MonitoringEvent monitoringEvent) {
		if (monitoringEvent.getTopic().contains(
				Topics.APPLICATIONS + "/General")) {

			ApplicationType[] jsdlAppArr = (ApplicationType[]) monitoringEvent
					.getDetailsArray();
			if (jsdlAppArr != null) {
				ComputingManagerT cmType = gDoc.getDomains()
						.getAdminDomainArray(0).getComputingServiceArray(0)
						.getComputingManagerArray(0);
				ApplicationEnvironments appEnvsType = cmType.getApplicationEnvironments();
				if (appEnvsType == null) {
					appEnvsType = cmType.addNewApplicationEnvironments();
				}

				for (int i = 0; i < jsdlAppArr.length; i++) {
					ApplicationEnvironmentT appEnv = appEnvsType
							.addNewApplicationEnvironment();

					// set the application id same as application name but
					// without whitespaces
					appEnv.setID(jsdlAppArr[i].getApplicationName().replaceAll(
							" ", "")
							+ "_" + jsdlAppArr[i].getApplicationVersion());
					jsdlAppArr[i].getApplicationVersion();
					appEnv.setName(jsdlAppArr[i].getApplicationName());
					appEnv.setAppVersion(jsdlAppArr[i].getApplicationVersion());
				}

			}
		}
	}

	private void setEndpointInfo(MonitoringEvent monitoringEvent, String unique) {
		if (monitoringEvent.getTopic().contains(Topics.SITE + "/General")) {
			SiteDocument sd = (SiteDocument) monitoringEvent.getDetails();
			ComputingServiceT csType = gDoc.getDomains().getAdminDomainArray(0)
					.getComputingServiceArray(0);
			ComputingEndpointT ceType = null;

			if(csType.getComputingEndpointArray().length>0){
				ceType = csType.getComputingEndpointArray(0);
			}else{
				ceType = csType.addNewComputingEndpoint();
			}
			
			if (ceType.getID() == null) {
				ceType.setID("urn:unicore:ce:"+unique);
			}
			if (ceType.getName() == null || ceType.getInterfaceName() == null) {
				if (sd.getSite().getName() != null) {
					ceType.setName(sd.getSite().getName());
					ceType.setInterfaceName(sd.getSite().getName());
				} else {
					ceType.setName("unknown");
					ceType.setInterfaceName("unknown");
				}
			}
			if (ceType.getQualityLevel() == null) {
				ceType.setQualityLevel(QualityLevelT.PRODUCTION);
			}

			if (ceType.getURL() == null) {
				if (sd.getSite().getURL() != null) {
					ceType.setURL(sd.getSite().getURL());
				} else {
					ceType.setURL("http://localhost");
				}

			}
			if (ceType.getHealthState() == null) {
				ceType.setHealthState(EndpointHealthStateT.OK);
			}
			if (ceType.getServingState() == null) {
				ceType.setServingState(ServingStateT.PRODUCTION);
			}

			if (ceType.getTechnology() == null) {
				ceType.setTechnology("webservice");
			}

			ceType.setBaseType(getBaseType("Endpoint"));
			setVersionInformation(ceType);
		}
	}

	private void setLocationInformation(String unique) {
		ComputingServiceT csT = gDoc.getDomains().getAdminDomainArray(0)
				.getComputingServiceArray(0);
		LocationT locT = csT.getLocation();
		if (locT == null) {
			locT = csT.addNewLocation();
			locT.setID("urn:unicore:loc:"+unique);
			try {
				JSONObject locObj = jsObj.getJSONObject("location");
				if (locObj != null) {
					locT.setLongitude(Float.valueOf(locObj
							.getString("longitude")));
					locT.setLatitude(Float.valueOf(locObj.getString("latitude")));
					locT.setCountry(locObj.getString("country"));
					locT.setPlace(locObj.getString("city"));
					locT.setName(locObj.getString("name"));
				}

			} catch (JSONException e) {
				logger.warn(
						"Error in reading 'location' property from cipjson file",
						e);
				// setting default location to Germany
				locT.setLongitude(52.0F);
				locT.setLatitude(3.0F);
				locT.setCountry("Germany");
				locT.setPlace("Jülich");
				locT.setName("FZJ Demo Site");
			}
		}

	}

	private void setVersionInformation(ComputingEndpointT endpoint){
		//add UNICORE specific version
		String uVersion=getClass().getPackage().getSpecificationVersion();
		if(uVersion==null)uVersion="DEVELOPMENT";
		endpoint.addInterfaceVersion(uVersion);
		endpoint.setImplementationName("UNICORE");
		endpoint.setImplementationVersion(uVersion);
		endpoint.setInterfaceName("UNICORE");
		
		//add EMI Version, if present
		File emiFile=new File(emiVersionFile);
		if(emiFile.exists() && emiFile.canRead()){
			try{
				String emiVersion=FileUtils.readFileToString(emiFile);
				endpoint.addOtherInfo("MiddlewareName=EMI");
				endpoint.addOtherInfo("MiddlewareVersion="+emiVersion);
			}catch(IOException ex){
				logger.trace("Can't read emi version.", ex);
			}
		}
	}
	
	private String readFileAsString(String filePath)throws java.io.IOException {
		return FileUtils.readFileToString(new File(filePath));
	}

	//for testing
	void setVersionFile(String fileName){
		emiVersionFile=fileName;
	}
	
}
