/*********************************************************************************
 * Copyright (c) 2007 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/

package de.fzj.unicore.cisprovider.impl;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;

import de.fzj.unicore.cisprovider.Monitor;
import de.fzj.unicore.cisprovider.MonitorEventListener;
import de.fzj.unicore.cisprovider.MonitoringEvent;
import de.fzj.unicore.cisprovider.Sensor;

/**
 * default implementation of the {@link Monitor} interface<br/>
 * 
 * @author schuller
 */
public class DefaultMonitor implements Monitor, MonitorEventListener{

	private static Logger logger=Logger.getLogger(DefaultMonitor.class.getName());
	
	public DefaultMonitor(){
	}
	
	private Set<Sensor> sensors=new HashSet<Sensor>();
	
	private Map<String,MonitoringEvent> events=new HashMap<String,MonitoringEvent>();
	
	public void addSensor(Sensor sensor) {
		sensor.addEventListener(this);
		sensors.add(sensor);
	}

	public void removeSensor(Sensor sensor) {
		sensors.remove(sensor);
	}

	public List<MonitoringEvent> getEvents(String topic, Calendar from, Calendar to, int limit) {
		List<MonitoringEvent> ev=new ArrayList<MonitoringEvent>();
		synchronized(events){
			int i=0;
			if(limit<=0)ev.addAll(events.values());
			else{
				for(MonitoringEvent e: events.values()){
					//filter by time
					if(to!=null && to.before(e.getTime())){
						continue;
					}
					if(from!=null && from.after(e.getTime())){
						continue;
					}
					
					//filter by topic
					if(e.matchesTopic(topic)){
						ev.add(e);
						i++;
					}
					if(i>=limit)break;
					
				}
			}
			//add last events from the sensors
			for(Sensor s: sensors){
				if(i>=limit)break;
				MonitoringEvent e=s.getLastEvent();
				if(e!=null && !ev.contains(e) && e.matchesTopic(topic)){
					ev.add(e);
				}
				i++;
			}
		}
		
		return ev;
	}

	public void notify(MonitoringEvent event) {
		synchronized(events){
			String id=event.getID();
			if(logger.isDebugEnabled()){
				logger.debug("Notification for event <"+id+"> on "+event.getTopic() );
			}
			
			String supersedes=event.supersedes();
			if(supersedes!=null){
				logger.debug("Removing <"+supersedes+"> superseded by <"+id+">");
				events.remove(supersedes);
			}
			
			//TODO need to limit the size here!!!
			//if(events.size()>MAX_EVENTS)....
			
			events.put(id,event);
		}
	}

}
