/*********************************************************************************
 * Copyright (c) 2007 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/

package de.fzj.unicore.cisprovider.impl;

import java.math.BigInteger;
import java.util.Calendar;
import java.util.List;

import org.apache.log4j.Logger;
import org.ogf.schemas.glue.x2009.x03.spec20R1.DomainsDocument;
import org.ogf.schemas.glue.x2009.x03.spec20R1.ServiceT;

import de.fzj.unicore.cisprovider.CIPProperties;
import de.fzj.unicore.cisprovider.CISInfoProvider;
import de.fzj.unicore.cisprovider.Monitor;
import de.fzj.unicore.cisprovider.MonitoringEvent;
import de.fzj.unicore.cisprovider.ServiceInfoGenerator;
import de.fzj.unicore.cisprovider.xmlbeans.GetGlueInfoRequestDocument;
import de.fzj.unicore.cisprovider.xmlbeans.GetGlueInfoResponseDocument;
import de.fzj.unicore.cisprovider.xmlbeans.GetGlueInfoResponseDocument.GetGlueInfoResponse;
import de.fzj.unicore.cisprovider.xmlbeans.GetInfoRequestDocument;
import de.fzj.unicore.cisprovider.xmlbeans.GetInfoResponseDocument;
import de.fzj.unicore.cisprovider.xmlbeans.GetServiceInfoRequestDocument;
import de.fzj.unicore.cisprovider.xmlbeans.GetServiceInfoResponseDocument;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.KernelInjectable;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import eu.unicore.util.Log;

/**
 * implementation of the {@link CISInfoProvider} interface
 * 
 * @author schuller
 */
public class CISInfoProviderImpl implements CISInfoProvider, KernelInjectable {

	static final Logger logger = Log.getLogger(Log.SERVICES,CISInfoProvider.class);

	private Kernel kernel;
	
	public void setKernel(Kernel kernel){
		this.kernel=kernel;
	}
	
	public GetInfoResponseDocument getInfo(GetInfoRequestDocument request) {
		try {
			CIPProperties cfg = kernel.getAttribute(CIPProperties.class);
			Monitor m = cfg.getMonitor();
			String topic = request.getGetInfoRequest().getTopic();
			Calendar from = request.getGetInfoRequest().getFrom();
			Calendar to = request.getGetInfoRequest().getTo();
			BigInteger requestedLimit = request.getGetInfoRequest()
					.getMaxNumberOfRecords();
			int limit = 100;
			if (requestedLimit != null) {
				limit = requestedLimit.intValue();
			}
			if (logger.isDebugEnabled()) {
				logger.debug("Request: topic=" + topic + ", from=" + from
						+ ", to=" + to + ", limit=" + limit);
			}
			List<MonitoringEvent> list = m.getEvents(topic, from, to, limit);
			GetInfoResponseDocument res = GetInfoResponseDocument.Factory
					.newInstance();
			res.addNewGetInfoResponse().setFeed(
					FeedGenerator.generateFeed(list, kernel).getFeed());
			if (logger.isDebugEnabled()) {
				logger.debug("Response: " + res);
			}
			return res;
		} catch (Exception e) {
			Log.logException("Error processing request.", e, logger);
			throw new RuntimeException(e);
		}
	}

	public GetGlueInfoResponseDocument getGlueInfo(
			GetGlueInfoRequestDocument req) throws BaseFault {
		
		CIPProperties cfg = kernel.getAttribute(CIPProperties.class);
		
		GetGlueInfoResponseDocument res = GetGlueInfoResponseDocument.Factory
				.newInstance();
		try {
			GetGlueInfoResponse resType = res.addNewGetGlueInfoResponse();
			resType.addNewDomains();
			String unique=kernel.getContainerProperties().getBaseUrl();
			DomainsDocument gDoc = new GlueInstanceGenerator().generate(cfg,unique);
			resType.setDomains(gDoc.getDomains());
				
		} catch (Exception e) {
			Log.logException("Error processing request.", e, logger);
			throw BaseFault.createFault("Error processing request.", e);
		}
		
		return res;
	}

	public GetServiceInfoResponseDocument getServiceInfo(
			GetServiceInfoRequestDocument req) throws BaseFault {
		GetServiceInfoResponseDocument res = GetServiceInfoResponseDocument.Factory.newInstance();
		ServiceT[] arrServiceT= ServiceInfoGenerator.getServiceInformation(kernel).toArray(new ServiceT[0]);
		res.addNewGetServiceInfoResponse().setServiceArray(arrServiceT);
		return res;
	}

}
