/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.BaseResource;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.grid.Job;
import eu.unicore.hila.grid.Site;

/**
 * @author bjoernh
 * 
 *         25.08.2009 15:43:43
 * 
 */
@ResourceType(locationStructure = { "unicore6:/storages/?",
	"unicore6:/{user}@storages/?", "unicore6:/sites/{siteName}/storages/?",
	"unicore6:/{user}@sites/{siteName}/storages/?" })
public class Unicore6StoragesCollection extends BaseResource {

    /**
     * @param location
     */
    public Unicore6StoragesCollection(Location location) {
	super(location);
    }

    /**
     * @see eu.unicore.hila.Resource#getChildren()
     */
    public List<Resource> getChildren() throws HiLAException {
	List<Resource> storages = new ArrayList<Resource>();

	Resource parent = this.getParent();
	if (parent instanceof Site) {
	    Site site = (Site) parent;
	    storages.addAll(site.getStorages());
	} else if (parent instanceof Job) {
	    Job task = (Job) parent;
	    storages.add(task.getWorkingDirectory().getStorage());
	} else if (parent instanceof Unicore6Grid) {
	    Unicore6Grid grid = (Unicore6Grid) parent;
	    storages.addAll(discoverGlobalStorages(grid));
	} else {
	    throw new HiLAException(
		    "Cannot determine what kind of storages you're looking for.");
	}

	return storages;
    }

    /**
     * @param grid
     * @return
     * @throws HiLAException 
     */
    private Collection<? extends Resource> discoverGlobalStorages(
	    Unicore6Grid grid) throws HiLAException {
	List<String> registryURLs = grid.getProperties().getRegistryUrls();
	return StorageLocator.getInstance().getAllStorages(registryURLs,
		grid.getProperties(), 5000, this.getLocation());
    }

}
