/*********************************************************************************
 * Copyright (c) 2009-2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;

import org.apache.log4j.Logger;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.DataStagingType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDefinitionDocument;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ResourcesDocument;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.SourceTargetType;
import org.unigrids.x2006.x04.services.sms.StoragePropertiesDocument;
import org.unigrids.x2006.x04.services.tss.ApplicationResourceType;
import org.unigrids.x2006.x04.services.tss.SubmitDocument;
import org.unigrids.x2006.x04.services.tss.TargetSystemPropertiesDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.client.JobClient;
import de.fzj.unicore.uas.client.ReservationClient;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.uas.client.TSFClient;
import de.fzj.unicore.uas.client.TSSClient;
import de.fzj.unicore.wsrflite.xfire.ClientException;
import eu.unicore.hila.Location;
import eu.unicore.hila.Metadata;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseSite;
import eu.unicore.hila.exceptions.HiLACannotContactSiteException;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLAFactoryException;
import eu.unicore.hila.exceptions.HiLALocationSyntaxException;
import eu.unicore.hila.exceptions.HiLANotImplementedException;
import eu.unicore.hila.exceptions.HiLANotSupportedException;
import eu.unicore.hila.exceptions.HiLAResourceAlreadyExistsException;
import eu.unicore.hila.exceptions.HiLAResourceNotFoundException;
import eu.unicore.hila.grid.File;
import eu.unicore.hila.grid.Job;
import eu.unicore.hila.grid.Reservation;
import eu.unicore.hila.grid.Site;
import eu.unicore.hila.grid.Storage;
import eu.unicore.hila.grid.resources.ResourceDescription;
import eu.unicore.hila.job.model.JobModel;
import eu.unicore.hila.job.model.MappingFactory;
import eu.unicore.hila.job.spi.MappingFactorySpi;
import eu.unicore.hila.job.spi.ModelToNative;
import eu.unicore.hila.job.spi.NoSuchMappingException;
import eu.unicore.util.httpclient.IClientConfiguration;

/**
 * @author bjoernh
 * 
 *         29.05.2009 13:41:35
 * 
 */
@ResourceType(locationStructure = { "unicore6:/sites/{siteName}/?",
	"unicore6:/{user}@sites/{siteName}/?" })
public class Unicore6Site extends BaseSite implements Site {
    /**
    * 
    */
   private static final String MSG_SITE_UNAVAILABLE = "Site is unavailable.";

   private static final Logger log = Logger.getLogger(Unicore6Site.class);

    private static final Cache sitesCache;
    static {
	final CacheManager manager = CacheManager.getInstance();
	manager.addCache(new Cache(Unicore6Site.class.getName(), 500, true,
		true, 0, 0));
	sitesCache = manager.getCache(Unicore6Site.class.getName());
    }

    private List<TSSClient> tssClients;

    private final Metadata metadata;

    private Unicore6ReservationsCollection reservationsCollection;

   private TSFClient tsfClient;

    private Unicore6Site(Location _location, Object... _extraInformation)
	    throws HiLAException {
	super(_location);
	for (Object object : _extraInformation) {
	    if (object instanceof List<?>
		    && ((List) object).get(0) instanceof TSSClient) {
		this.tssClients = (List<TSSClient>) object;
         } else if (object instanceof TSFClient) {
            this.tsfClient = (TSFClient) object;
	    }
	}
	this.metadata = new Metadata();
	if (this.tssClients == null || this.tssClients.size() == 0) {
	    throw new HiLACannotContactSiteException(
		    "Missing TSS for target system.");
	}
    }

    public static Unicore6Site locate(Location _location,
	    Object... extraInformation) throws HiLAException {

	Element cacheElement = sitesCache.get(_location);
		if (cacheElement != null) {
			if (log.isDebugEnabled()) {
				log.debug("Found cache entry for " + _location);
			}
			Unicore6Site site = (Unicore6Site) cacheElement.getObjectValue();
			if (site.ok()) {
				return site;
			} else {
				sitesCache.removeElement(cacheElement);
			}
		}

	if (log.isDebugEnabled()) {
	    log.debug("Cache for " + _location + " was invalid.");
	}

	if (extraInformation.length > 0) {
	    if (log.isDebugEnabled()) {
		log.debug("Creating new Site object for " + _location);
		if (log.isTraceEnabled()) {
		    for (Object object : extraInformation) {
			log.trace("Using extraInformation " + object);
		    }
		}
	    }
	    final Unicore6Site site = new Unicore6Site(_location,
		    extraInformation);
	    sitesCache.put(new Element(_location, site));
	    return site;
	}

	Location sitesLoc = findParentLocationOfType(
		Unicore6SitesCollection.class, _location, Unicore6Grid.class);
	for (Resource res : sitesLoc.locate().getChildren()) {
	    if (res.getLocation().equals(_location)) {
		return (Unicore6Site) res;
	    }
	}

	throw new HiLAResourceNotFoundException("Site " + _location
		+ " cannot be found.");
    }

    /**
     * Add reservations to children, as they're not returned by {@link BaseSite}
     * by default.
     * 
     * @see eu.unicore.hila.common.grid.BaseSite#getChildren()
     * @since 2.2
     */
    @Override
    public synchronized java.util.List<eu.unicore.hila.Resource> getChildren()
	    throws HiLALocationSyntaxException {
	List<Resource> children = super.getChildren();
	if (this.reservationsCollection == null) {
	    this.reservationsCollection = new Unicore6ReservationsCollection(
		    this.getLocation().getChildLocation("reservations"));
	}
	if (!children.contains(this.reservationsCollection)) {
	    children.add(this.reservationsCollection);
	}
	return children;
    };

    /**
     * @see eu.unicore.hila.common.BaseResource#getMetadata()
     */
    @Override
    public Metadata getMetadata() throws HiLAException {
	updateMetadata();

	return this.metadata;
    }

   /**
    * @throws HiLAException
    * @see eu.unicore.hila.grid.Submittable#getTask(java.lang.String)
    */
   public Job getTask(String taskId) throws HiLAException {
      if (!ok()) {
         throw new HiLACannotContactSiteException(MSG_SITE_UNAVAILABLE);
      }
      String baseURL = tssClients.get(0).getEPR().getAddress().getStringValue()
            .replaceAll("TargetSystemService.*", "");
      String jmAddress = baseURL + "JobManagement?res=" + taskId;
      EndpointReferenceType epr = EndpointReferenceType.Factory.newInstance();
      epr.addNewAddress().setStringValue(jmAddress);
      return createJobFromEPR(epr);
   }

   public Job submit(JobModel _jm) throws HiLAException {
      MappingFactorySpi mapping;
      try {
         mapping = MappingFactory.getMapping("jsdl");
         ModelToNative m2n = mapping.createModelToNativeMapping(_jm);
         return submit((JobDefinitionDocument) m2n.getNative());
      } catch (NoSuchMappingException e) {
         throw new HiLAException("Unable to map job model to JSDL.", e);
      }

   }

    /**
     * @throws HiLANotImplementedException
     * @see eu.unicore.hila.grid.Submittable#submit(eu.unicore.hila.grid.job.JobDescription)
     */
    public Job submit(JobDefinitionDocument jdd) throws HiLAException {
      if (!ok()) {
         throw new HiLACannotContactSiteException(MSG_SITE_UNAVAILABLE);
      }
	DataStagingType[] stagings = jdd.getJobDefinition().getJobDescription()
		.getDataStagingArray();
	for (DataStagingType staging : stagings) {
	    SourceTargetType src = staging.getSource();
	    if (src != null) {
		Location srcLoc = new Location(src.getURI());
		if (srcLoc.isLocationOfType(File.class)) {
		    try {
			final Unicore6File f = (Unicore6File) srcLoc.locate();
			src.setURI("BFT:".concat(f.getStagingURI()));
		    } catch (ClassCastException e) {
			// don't do anything
		    }
		}
	    }
	    SourceTargetType tgt = staging.getTarget();
	    if (tgt != null) {
		Location tgtLoc = new Location(tgt.getURI());
		if (tgtLoc.isLocationOfType(File.class)) {
		    try {
			final Unicore6File f = (Unicore6File) tgtLoc.locate();
			tgt.setURI("BFT:".concat(f.getStagingURI()));
		    } catch (ClassCastException e) {
			// don't do anything
		    }
		}
	    }
	}

	SubmitDocument sd = SubmitDocument.Factory.newInstance();
	sd.addNewSubmit().setJobDefinition(jdd.getJobDefinition());

	try {
			TSSClient existingTssClient = tssClients.get(0);
            Unicore6SecurityProperties sp = (Unicore6SecurityProperties) existingTssClient
                    .getSecurityConfiguration().clone();
			sp.getETDSettings().setExtendTrustDelegation(true);
			TSSClient tssClient = new TSSClient(existingTssClient.getEPR(), sp);
			JobClient jobClient = tssClient.submit(sd);
	    String taskId = epr2TaskId(jobClient.getEPR());
	    Job task = new Unicore6Job(location.getChildLocation("tasks")
		    .getChildLocation(taskId), jobClient);
	    return task;
	} catch (Exception e) {
	    throw new HiLAException("Error submitting job to target system.", e);
	}
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseSite#getStorages()
     */
    @Override
    public List<Storage> getStorages() throws HiLAException {
	return getStorages1();
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseSite#getStorages()
     */
    public List<Storage> getStorages1() throws HiLAException {
	List<Storage> storages = new ArrayList<Storage>();

      if (!ok()) {
         throw new HiLACannotContactSiteException(MSG_SITE_UNAVAILABLE);
      }
	List<EndpointReferenceType> storageReferences = tssClients.get(0)
		.getStorages();
	for (EndpointReferenceType storageEPR : storageReferences) {
	    try {
		// introduce a res_id -> name map to avoid this extra call for
		// already existing storages
                IClientConfiguration storSecProp = this.tssClients.get(0)
                        .getSecurityConfiguration().clone();
		StorageClient sClient = new StorageClient(storageEPR
			.getAddress().getStringValue(), storageEPR, storSecProp);
		StoragePropertiesDocument spd = sClient
			.getResourcePropertiesDocument();
		String fileSystemName = spd.getStorageProperties()
			.getFileSystem().getName();

		Location storageLocation = this.location.getChildLocation(
			"storages").getChildLocation(fileSystemName);
		Storage storage = (Storage) storageLocation.locate(sClient);

		storages.add(storage);
	    } catch (CloneNotSupportedException e) {
		throw new HiLAException("Unable to clone security properties.",
			e);
	    } catch (Exception e) {
		throw new HiLAException(
			"Unexpected exception while retrieving list of storages.",
			e);
	    }
	}

	return storages;
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.common.grid.BaseSite#getStorage(java.lang.String)
     */
    public Storage getStorage(String storageName) throws HiLAException {
	List<Storage> storages = getStorages();
	for (Storage storage : storages) {
	    if (storage.getName().equals(storageName)) {
		return storage;
	    }
	}
	throw new HiLAException("Storage does not exist.");
    }

    /**
    * @throws HiLAFactoryException
    * @throws HiLACannotContactSiteException
    * @see eu.unicore.hila.grid.Submittable#getTasks()
    */
   public List<Job> getTasks() throws HiLAFactoryException,
         HiLACannotContactSiteException {
	final List<Job> tasks = new ArrayList<Job>();

      if (!ok()) {
         throw new HiLACannotContactSiteException(MSG_SITE_UNAVAILABLE);
      }
	for (TSSClient tssClient : tssClients) {
		try {
		    List<EndpointReferenceType> jobEprs = tssClient.getJobs();
		    for (EndpointReferenceType jobEpr : jobEprs) {
			tasks.add(createJobFromEPR(jobEpr));
		    }
		} catch (Exception e) {
		    log.error("Unable to list jobs.", e);
            ok(); // checks connection and drops TSSClient if no ok.
		}
	}

	return tasks;
    }

    /**
     * @param tssClient
     * @param jobEpr
     * @throws HiLAException
     */
    private Job createJobFromEPR(EndpointReferenceType jobEpr)
	    throws HiLAException {
      if (!ok()) {
         throw new HiLACannotContactSiteException(MSG_SITE_UNAVAILABLE);
      }
	String resId = epr2TaskId(jobEpr);
	try {
	    Location taskLocation = location
		    .getChildLocation(TasksCollection.TASKS.concat("/").concat(
			    resId));
	    Job task = null;
	    try {
		task = new Unicore6Job(taskLocation, new JobClient(jobEpr
			.getAddress().getStringValue(), jobEpr, tssClients
                        .get(0).getSecurityConfiguration().clone()));
	    } catch (HiLAResourceAlreadyExistsException e) {
		task = (Job) location.getChildLocation(
			TasksCollection.TASKS.concat("/").concat(resId))
			.locate();
	    }
	    if (log.isDebugEnabled()) {
		log.debug("Located task ".concat(task.getLocation().toString()));
	    }
	    return task;
	} catch (CloneNotSupportedException e) {
	    throw new HiLAException("Unable to clone security properties.", e);
	} catch (Exception e) {
	    throw new HiLAException(
		    "Unexpected exception while retrieving list of tasks.", e);
	}
    }

    /**
     * TODO determine Task ID from ReferenceParameters
     * 
     * @param jobEpr
     * @return
     */
    public static String epr2TaskId(EndpointReferenceType jobEpr) {
	String resId = jobEpr.getAddress().getStringValue()
		.replaceFirst(".*\\?res=", "");
	if (log.isDebugEnabled()) {
	    log.debug("resId = " + resId);
	}
	return resId;
    }

    /**
     * @throws HiLAException
     * 
     */
    private void updateMetadata() throws HiLAException {
      if (!ok()) {
         throw new HiLACannotContactSiteException(MSG_SITE_UNAVAILABLE);
      }
	try {
	    final TargetSystemPropertiesDocument tsp = tssClients.get(0)
		    .getResourcePropertiesDocument();

	    // applications
	    final List<ApplicationResourceType> applications = new ArrayList<ApplicationResourceType>();
	    for (ApplicationResourceType application : tsp
		    .getTargetSystemProperties().getApplicationResourceArray()) {
		applications.add(application);
	    }
	    metadata.setData("applications", applications);

	    metadata.setData("resourceProperties", tsp);

	} catch (Exception e) {
	    throw new HiLAException(
		    "Unable to retrieve TargetSystemProperties.", e);
	}

    }

    /**
     * @see eu.unicore.hila.grid.Reservable#reserve(ResourceDescription,
     *      Calendar)
     */
    @Override
    public Reservation reserve(ResourceDescription _rd, Calendar _startTime)
	    throws HiLAException {
      if (!ok()) {
         throw new HiLACannotContactSiteException(MSG_SITE_UNAVAILABLE);
      }
	try {
	    if (!tssClients.get(0).supportsReservation()) {
		throw new HiLANotSupportedException(
			"Server does not support reservation.");
	    }
      } catch (HiLANotSupportedException e) {
         // this exception goes out directly
         throw e;
      } catch (Exception e) {
		throw new HiLAException(
			"Cannot determine whether server supports reservation.",
			e);
	}

	ReservationClient resClient = null;
	if (_rd.getActualDescription() instanceof ResourcesDocument) {
	    try {
		resClient = tssClients.get(0).createReservationClient(
			(ResourcesDocument) _rd.getActualDescription(),
			_startTime);
	    } catch (Exception e) {
		throw new HiLAException("Unable to create reservation.", e);
	    }
	} else {
	    throw new HiLAException("Unknown resource description.");
	}

	if (resClient != null) {
	    try {
		Location resLoc = getLocation()
			.getChildLocation("reservations").getChildLocation(
				epr2TaskId(resClient.getEPR()));
		Unicore6Reservation u6res = (Unicore6Reservation) resLoc
			.locate(resClient);
		return u6res;
	    } catch (Exception e) {
		throw new HiLAException("Cannot get reservation reference.", e);
	    }

	} else {
	    throw new HiLAException("Unable to create reservation.");
	}
    }

    /**
     * @see eu.unicore.hila.grid.Reservable#getReservations()
     * @since 2.2
     */
    @Override
    public List<Reservation> getReservations() throws HiLAException {
	List<Reservation> reservations = new ArrayList<Reservation>();
      if (!ok()) {
         throw new HiLACannotContactSiteException(MSG_SITE_UNAVAILABLE);
      }
	for (TSSClient tssClient : tssClients) {

	    List<EndpointReferenceType> resEPRs = tssClient.getReservations();

	    for (EndpointReferenceType resEPR : resEPRs) {
		ReservationClient resClient;
		try {
		    resClient = new ReservationClient(resEPR.getAddress()
			    .getStringValue(), resEPR, tssClient
                            .getSecurityConfiguration().clone());
		} catch (CloneNotSupportedException e) {
		    throw new HiLAException(
			    "Something weird with security setup.", e);
		} catch (Exception e) {
		    throw new HiLAException(
			    "Cannot create reservation client.", e);
		}
		Location resLoc = getLocation()
			.getChildLocation("reservations").getChildLocation(
				epr2TaskId(resEPR));
		reservations.add((Reservation) resLoc.locate(resClient));
	    }
	}
	return reservations;
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.Reservable#getReservation(java.lang.String)
     * @since 2.2
     */
    @Override
    public Reservation getReservation(String _name) throws HiLAException {
	// TODO do analogous optimization as for tasks here
	List<Reservation> reservations = getReservations();
	for (Reservation reservation : reservations) {
	    if (reservation.getName().equals(_name)) {
		return reservation;
	    }
	}
	return null;

    }

    /**
     * @see eu.unicore.hila.common.BaseResource#ok()
     */
    @Override
    public boolean ok() {
	List<TSSClient> removeList = new ArrayList<TSSClient>();
	for (TSSClient tssClient : tssClients) {
	    try {
		if (tssClient.checkConnection()) {
		    return true;
		}
	    } catch (ClientException e) {
	    } finally {
		removeList.add(tssClient);
	    }
	}
	tssClients.removeAll(removeList);
      // as a last resort, try and create a new TSS resource
      try {
         tssClients.add(tsfClient.createTSS());
         return true;
      } catch (Exception e) {
         return false;
      }
    }

}
