/*********************************************************************************
 * Copyright (c) 2010-2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Callable;

import de.fzj.unicore.uas.client.FileTransferClient;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.wsrflite.xfire.ClientException;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceNotDestroyedFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnavailableFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnknownFault;
import eu.unicore.hila.HiLA;
import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseSimpleTransfer;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLANotImplementedException;
import eu.unicore.hila.exceptions.HiLAResourceNotFoundException;
import eu.unicore.hila.grid.SimpleTransfer;
import eu.unicore.hila.grid.TaskStatus;
import eu.unicore.hila.grid.unicore6.util.FileTransfers;

/**
 * @author bjoernh
 * 
 *         12.03.2010 12:39:39
 * 
 */
@ResourceType(locationStructure = {
	"unicore6:/sites/{site}/storages/{storage}/imports/{import}",
	"unicore6:/{user}@sites/{site}/storages/{storage}/imports/{import}",
	"unicore6:/sites/{site}/tasks/{task}/wd/imports/{import}",
	"unicore6:/{user}@sites/{site}/tasks/{task}/wd/imports/{import}",
	"unicore6:/storages/{storage}/imports/{import}",
	"unicore6:/{user}@storages/{storage}/imports/{import}" })
public class Unicore6ImportTask extends BaseSimpleTransfer implements
	SimpleTransfer, Callable<TaskStatus> {

    private StorageClient storageClient;
    private String path;
    private eu.unicore.hila.grid.File remoteFile;
    private File localFile;
    private boolean overwrite;
    private FileTransferClient importClient;
    private long maxBytes;
    private long currentBytes;

    /**
     * @param _storageLocation
     * @param _storageClient
     * @param _path
     * @param _localFile
     * @param _overwrite
     * @throws HiLAException
     */
    public Unicore6ImportTask(Location _storageLocation,
	    StorageClient _storageClient, String _path,
	    eu.unicore.hila.grid.File _remoteFile, File _localFile,
	    boolean _overwrite) throws HiLAException {
	super(_storageLocation.getChildLocation("imports").getChildLocation(
		UUID.randomUUID().toString()));

	this.storageClient = _storageClient;
	this.path = _path;
	this.remoteFile = _remoteFile;
	this.localFile = _localFile;
	this.overwrite = _overwrite;

	setCurrentState(TaskStatus.PENDING);

      HiLA.getExecutor().submit(this);

	Unicore6ImportsCollection ic = (Unicore6ImportsCollection) this
		.getParent();
	if (ic != null) {
	    ic.addTask(this);
	}
    }

    /**
     * @param _location
     */
    protected Unicore6ImportTask(Location _location) {
	super(_location);
    }

    public static synchronized Unicore6ImportTask locate(Location _location,
	    Object... _extrainformation) throws HiLAException {
	Resource importsRes = _location.getParentLocation().locate();
	List<Resource> imports = importsRes.getChildren();
	for (Resource resource : imports) {
	    if (resource.getLocation().equals(_location)) {
		return (Unicore6ImportTask) resource;
	    }
	}
	throw new HiLAResourceNotFoundException("This Import does not exist: "
		+ _location);
    }

    /**
     * @see java.util.concurrent.Callable#call()
     */
    @Override
    public TaskStatus call() throws Exception {
	setCurrentState(TaskStatus.RUNNING);
	if (!this.localFile.exists()) {
	    HiLAException e = new HiLAException("No such file or directory.");
	    setCurrentState(TaskStatus.FAILED, e);
	    throw e;
	}
	if (this.storageClient != null) {
	    InputStream fis = null;
	    try {
            Map<String, String> extraParameters = new HashMap<String, String>();
            FileTransfers.getDefaultUftpParameters(extraParameters);
            this.importClient = this.storageClient.getImport(this.path, false,
                  extraParameters, FileTransfers.getFTPrefs());
		if (this.overwrite || !this.remoteFile.exists()) {
		    fis = new FileInputStream(this.localFile);
		    importClient.writeAllData(fis);
		    fis.close();
		} else {
		    HiLAException e = new HiLAException(
			    "File exists, but may not be overwritten.");
		    setCurrentState(TaskStatus.FAILED, e);
		    importClient.destroy();
		    importClient = null;
		    throw e;
		}
	    } catch (IOException e) {
		HiLAException e2 = new HiLAException(
			"Unable to import from local file.", e);
		setCurrentState(TaskStatus.FAILED, e2);
		if (importClient != null) {
		    importClient.destroy();
		    importClient = null;
		}
		throw e2;
	    } catch (Exception e) {
		HiLAException e2 = new HiLAException(
			"Unable to write data to remote file.", e);
		setCurrentState(TaskStatus.FAILED, e2);
		if (importClient != null) {
		    importClient.destroy();
		    importClient = null;
		}
		throw e2;
	    } finally {
		if (fis != null) {
		    try {
			fis.close();
		    } catch (IOException e) {
			// at least we've tried
		    }
		}
	    }
	}
	setCurrentState(TaskStatus.SUCCESSFUL);
	importClient.destroy();
	importClient = null;
	return status();
    }

    /**
     * @see eu.unicore.hila.grid.Task#abort()
     */
    @Override
    public synchronized void abort() throws HiLAException {
	try {
	    if (importClient != null) {
		importClient.destroy();
		importClient = null;
	    }
	} catch (BaseFault e) {
	    throw new HiLAException(e.getMessage(), e);
	} catch (ResourceUnavailableFault e) {
	    throw new HiLAException(e.getMessage(), e);
	} catch (ResourceUnknownFault e) {
	    throw new HiLAException(e.getMessage(), e);
	} catch (ResourceNotDestroyedFault e) {
	    throw new HiLAException(e.getMessage(), e);
	} catch (ClientException e) {
	    throw new HiLAException(e.getMessage(), e);
	}
    }

    /**
     * @see eu.unicore.hila.grid.Task#getId()
     */
    @Override
    public String getId() {
	return this.location.getName();
    }

    /**
     * @see eu.unicore.hila.Resource#getChildren()
     */
    @Override
    public List<Resource> getChildren() throws HiLAException {
	final List<Resource> children = new ArrayList<Resource>();
	return children;
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseTask#getUnit()
     */
    @Override
    public String getUnit() throws HiLANotImplementedException {
	return "bytes";
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.common.grid.BaseTask#getMax()
     */
    @Override
    public synchronized long getMax() throws HiLAException {
	if (importClient != null) {
	    this.maxBytes = localFile.length();
	}
	return this.maxBytes;
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseTask#getCurrent()
     */
    @Override
    public synchronized long getCurrent() throws HiLAException {
	if (importClient != null) {
	    try {
		this.currentBytes = importClient.getTransferredBytes();
	    } catch (Exception e) {
		throw new HiLAException(
			"Unable to determine transfer progress.", e);
	    }
	}
	return this.currentBytes;
    }

}
