/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.unigrids.services.atomic.types.GridFileType;
import org.unigrids.services.atomic.types.PermissionsType;

import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import eu.unicore.hila.Location;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseFile;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.grid.File;
import eu.unicore.hila.grid.Grid;
import eu.unicore.hila.grid.SimpleTransfer;
import eu.unicore.hila.grid.Storage;
import eu.unicore.hila.grid.ThirdPartyTransfer;

/**
 * @author bjoernh
 * 
 *         27.08.2009 10:04:13
 * 
 */
@ResourceType(locationStructure = {
	"unicore6:/sites/{site}/storages/{storage}/files((/?)|(/.*))",
	"unicore6:/{user}@sites/{site}/storages/{storage}/files((/?)|(/.*))",
	"unicore6:/sites/{site}/tasks/{task}/wd/files((/?)|(/.*))",
	"unicore6:/{user}@sites/{site}/tasks/{task}/wd/files((/?)|(/.*))",
	"unicore6:/storages/{storage}/files((/?)|(/.*))",
	"unicore6:/{user}@storages/{storage}/files((/?)|(/.*))" })
public class Unicore6File extends BaseFile implements File {

    private static final Logger log = Logger.getLogger(Unicore6File.class);

   // TODO use EHCache for this
    private static final Map<Location, Unicore6File> fileMap = new HashMap<Location, Unicore6File>();

    private static final CharSequence REASON_FILENOTFOUND = "Reason: java.io.FileNotFoundException";
    private StorageClient storageClient = null;
    private Unicore6Storage storage = null;
    private String decodedPath = null;

    private GridFileType gridFile = null;
    private long gridFileTimeStamp = 0;

    /**
     * @param gridFile2
     * @throws HiLAException
     * 
     */
    private Unicore6File(Location _location, StorageClient _storageClient,
	    GridFileType _gridFile) throws HiLAException {
	super(_location);
	this.storageClient = _storageClient;
	this.gridFile = _gridFile;
	gridFileTimeStamp = System.currentTimeMillis();
	Location storageLocation = findParentLocationOfType(
		Unicore6Storage.class, _location, Unicore6Grid.class);
	try {
	    this.storage = (Unicore6Storage) storageLocation.locate();
	    if (this.storageClient == null) {
		this.storageClient = this.storage.getStorageClient();
	    }
	} catch (HiLAException e) {
	    // this is ok, but the file will not be usable
	    log.error("Couldn't locate storage");
	}
	this.path = storageLocation.getChildLocation("files").relativePath(
		_location);
	this.decodedPath = decode(this.path);
    }

    public static synchronized Unicore6File locate(Location _location,
	    Object... _extraInformation) throws HiLAException {
	if (fileMap.containsKey(_location)) {
	    Unicore6File file = fileMap.get(_location);
	    // An implicit update for performance reasons
	    for (Object object : _extraInformation) {
		if (object instanceof GridFileType) {
		    file.gridFile = (GridFileType) object;
		    file.gridFileTimeStamp = System.currentTimeMillis();
		}
	    }
	    return file;
	} else {
	    StorageClient storageClient = null;
	    GridFileType gridFile = null;
	    for (Object object : _extraInformation) {
		if (object instanceof StorageClient) {
		    storageClient = (StorageClient) object;
		}
		if (object instanceof GridFileType) {
		    gridFile = (GridFileType) object;
		}
	    }
	    Unicore6File file = new Unicore6File(_location, storageClient,
		    gridFile);
	    fileMap.put(_location, file);
	    return file;
	}
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.File#delete()
     */
    public boolean delete(boolean _recursive) throws HiLAException {
	if (this.isDirectory() && !_recursive) {
	    throw new HiLAException(
		    "Cannot delete directory without recursion.");
	}
	if (storageClient != null) {
	    try {
		storageClient.delete(decodedPath);
		gridFileTimeStamp = 0;
		return true;
	    } catch (BaseFault e) {
		if (e.getMessage().contains(REASON_FILENOTFOUND)) {
		    throw new HiLAException("File not found.", e);
		}
		throw new HiLAException("Error while deleting file.");
	    }
	}
	return false;
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.File#exists()
     */
    public boolean exists() throws HiLAException {
	if (storageClient == null) {
	    if (storage != null) {
		storageClient = storage.getStorageClient();
	    } else {
		throw new HiLAException("No access to storage.");
	    }
	}

	if (storageClient != null) {
	    try {
		this.gridFile = storageClient.listProperties(this.decodedPath);
		return true;
	    } catch (BaseFault e) {
		if (e.getMessage().contains(REASON_FILENOTFOUND)) {
		    return false;
		} else {
		    throw new HiLAException(
			    "Couldn't determine whether file exists.", e);
		}
	    } catch (FileNotFoundException e) {
		return false;
	    }
	}

	return false;
    }

    /**
     * @see eu.unicore.hila.grid.File#size()
     */
    public long size() throws HiLAException {
	updateGridFile();
	return this.gridFile.getSize();
    }

    /**
     * @see eu.unicore.hila.grid.File#getStorage()
     */
    public Storage getStorage() {
	return storage;
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.File#importFromLocalFile(java.io.File)
     */
    public SimpleTransfer importFromLocalFile(java.io.File localFile,
	    boolean _overwrite, boolean _recursive) throws HiLAException {
	Location storageLocation = findParentLocationOfType(Storage.class,
		this.location, Grid.class);
	Unicore6File importFile = null;
	if (this.exists() && this.isDirectory()) {
	    importFile = (Unicore6File) getChild(localFile.getName());
	} else {
	    importFile = this;
	}

	SimpleTransfer _import;
	if (localFile.isDirectory() && _recursive) {
	    importFile.mkdir();
         _import = new Unicore6RecursiveImportTask(localFile, importFile);
	} else {
	    _import = new Unicore6ImportTask(storageLocation,
		    importFile.storageClient, importFile.decodedPath,
		    importFile, localFile, _overwrite);
	}

	return _import;
    }

    /**
     * @see eu.unicore.hila.grid.File#exportToLocalFile(java.io.File)
     */
    public SimpleTransfer exportToLocalFile(java.io.File localFile,
	    boolean _overwrite, boolean _recursive) throws HiLAException {
	Location storageLocation = findParentLocationOfType(Storage.class,
		this.location, Grid.class);
	if (localFile.exists() && localFile.isDirectory()) {
	    localFile = new java.io.File(localFile, this.getName());
	}

	SimpleTransfer export;
	if (this.isDirectory() && _recursive) {
         export = new Unicore6RecursiveExportTask(this, localFile);
	} else {
	    export = new Unicore6ExportTask(storageLocation,
		    this.storageClient, this.decodedPath, this, localFile,
		    _overwrite);
	}

	return export;
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.File#isDirectory()
     */
    public boolean isDirectory() throws HiLAException {
	updateGridFile();
	return gridFile.getIsDirectory();
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.File#ls()
     */
    public List<File> ls() throws HiLAException {
	if (storageClient == null) {
	    if (storage != null) {
		storageClient = storage.getStorageClient();
	    } else {
		throw new HiLAException("No access to storage.");
	    }
	}
	if (this.isDirectory()) {
	    List<File> children = new ArrayList<File>();
	    try {
		GridFileType[] gridFiles = storageClient
			.listDirectory(this.decodedPath);
		// locate them.
		for (GridFileType gridFileType : gridFiles) {

		    String childPath = gridFileType.getPath();
		    if (childPath.startsWith("/")) {
			childPath = childPath.substring(1);
		    }
		    final File file = (File) storage.getLocation()
			    .getChildLocation("files")
			    .getChildLocation(encode(childPath))
			    .locate(storage.getStorageClient(), gridFileType);
		    children.add(file);

		}
		return children;
	    } catch (BaseFault e) {
		throw new HiLAException("Cannot list directory", e);
	    }
	} else {
	    // if this is not a directory, then return the file itself
	    return Collections.singletonList((File) this);
	}
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.File#mkdir()
     */
    public boolean mkdir(boolean _createParents) throws HiLAException {
	if (_createParents) {
	    try {
		storageClient.createDirectory(decodedPath);
		if (exists()) {
		    return true;
		}
		return false;
	    } catch (BaseFault e) {
		return false;
	    }
	} else {
	    File parent = (File) this.getParent();
	    if (parent.exists() && parent.isDirectory()) {
		try {
		    storageClient.createDirectory(decodedPath);
		    if (this.exists()) {
			return true;
		    }
		    return false;
		} catch (BaseFault e) {
		    return false;
		}
	    } else {
		return false;
	    }
	}
    }

    private String encode(String _original) {
	_original = _original.replace("#", "%23");
	_original = _original.replace("%", "%25");
	return _original.replace(" ", "%20");
    }

    private String decode(String _original) {
	_original = _original.replace("%23", "#");
	_original = _original.replace("%25", "%");
	return _original.replace("%20", " ");
    }

    /**
     * @see eu.unicore.hila.grid.File#isExecutable()
     */
    public boolean isExecutable() throws HiLAException {
	updateGridFile();
	return this.gridFile.getPermissions().getExecutable();
    }

    /**
     * @see eu.unicore.hila.grid.File#isReadable()
     */
    public boolean isReadable() throws HiLAException {
	updateGridFile();
	return this.gridFile.getPermissions().getReadable();
    }

    /**
     * @see eu.unicore.hila.grid.File#isWritable()
     */
    public boolean isWritable() throws HiLAException {
	updateGridFile();
	return this.gridFile.getPermissions().getWritable();
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.File#setIsExecutable(boolean)
     */
    public void setIsExecutable(boolean executable) throws HiLAException {
	updateGridFile();
	try {
	    PermissionsType perms = gridFile.getPermissions();
	    perms.setExecutable(executable);
	    storageClient.changePermissions(decodedPath, perms.getReadable(),
		    perms.getWritable(), perms.getExecutable());
	} catch (BaseFault e) {
	    throw new HiLAException("Unable to change executable permission", e);
	}
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.File#setIsReadable(boolean)
     */
    public void setIsReadable(boolean readable) throws HiLAException {
	updateGridFile();
	try {
	    PermissionsType perms = gridFile.getPermissions();
	    perms.setReadable(readable);
	    storageClient.changePermissions(decodedPath, perms.getReadable(),
		    perms.getWritable(), perms.getExecutable());
	} catch (BaseFault e) {
	    throw new HiLAException("Unable to change readable permission", e);
	}
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.File#setIsWritable(boolean)
     */
    public void setIsWritable(boolean writable) throws HiLAException {
	updateGridFile();
	try {
	    PermissionsType perms = gridFile.getPermissions();
	    perms.setWritable(writable);
	    storageClient.changePermissions(decodedPath, perms.getReadable(),
		    perms.getWritable(), perms.getExecutable());
	} catch (BaseFault e) {
	    throw new HiLAException("Unable to change executable permission", e);
	}
    }

    /**
     * @see eu.unicore.hila.grid.File#copyTo(eu.unicore.hila.grid.File, boolean)
     */
    @Override
    public void copyTo(File other, boolean overwrite, boolean _recursive)
	    throws HiLAException {
	if (storageClient == null) {
	    if (storage != null) {
		storageClient = storage.getStorageClient();
	    } else {
		throw new HiLAException("No access to storage.");
	    }
	}
	Unicore6File u6other;
	if (other instanceof Unicore6File) {
	    u6other = (Unicore6File) other;
	} else {
	    throw new HiLAException("Cannot copy to File of different type.");
	}

	if (!this.isDirectory()) {
	    if (u6other.exists() && u6other.isDirectory()) {
		u6other = (Unicore6File) u6other.getChild(this.getName());
	    } else if (u6other.exists() && !overwrite) {
		throw new HiLAException("File exists.");
	    }
	    if (this.getStorage().equals(other.getStorage())) {
		try {
		    storageClient.copy(this.decodedPath, u6other.decodedPath);
		} catch (BaseFault e) {
		    throw new HiLAException(
			    "Unable to copy File within storage.", e);
		}
	    } else {
		throw new HiLAException(
			"It is only possible to copy files within a storage."
				+ " Use transfer for different storages");
	    }
	} else { // recursive copy necessary
	    if (!_recursive) {
		throw new HiLAException(
			"Missing 'recursive' option to copy directory structures.");
	    } else {
		if (u6other.exists() && u6other.isDirectory()) {
		    u6other = (Unicore6File) u6other.getChild(this.getName());
		    if (this.isDirectory() && !u6other.exists()) {
			u6other.mkdir();
		    }
		} else if (u6other.exists() && !u6other.isDirectory()) {
		    throw new HiLAException(
			    "Cannot copy directory structure into a single file.");
		} else { // other directory does not exist and needs to be
			 // created
		    u6other.mkdir();
		}

		List<File> children = this.ls();
		for (File file : children) {
		    file.copyTo(
			    (Unicore6File) u6other.getChild(file.getName()),
			    overwrite, _recursive);
		}
	    }
	}
    }

    /**
     * @see eu.unicore.hila.grid.File#lastModified()
     */
    @Override
    public Date lastModified() throws HiLAException {
	updateGridFile();
	return this.gridFile.getLastModified().getTime();
    }

    /**
     * @see eu.unicore.hila.grid.File#moveTo(eu.unicore.hila.grid.File, boolean)
     */
    @Override
    public void moveTo(File other, boolean overwrite) throws HiLAException {
	if (storageClient == null) {
	    if (storage != null) {
		storageClient = storage.getStorageClient();
	    } else {
		throw new HiLAException("No access to storage.");
	    }
	}
	Unicore6File u6other;
	if (other instanceof Unicore6File) {
	    u6other = (Unicore6File) other;
	} else {
	    throw new HiLAException("Can only move files of the same type.");
	}
	if (u6other.exists() && u6other.isDirectory()) {
	    u6other = (Unicore6File) u6other.getChild(this.getName());
	} else if (u6other.exists() && !overwrite) {
	    throw new HiLAException("File exists.");
	}
	if (this.getStorage().equals(u6other.getStorage())) {
	    try {
		storageClient.rename(this.decodedPath, u6other.decodedPath);
		gridFileTimeStamp = 0;
	    } catch (BaseFault e) {
		throw new HiLAException("Unable to move file within storage.",
			e);
	    }
	} else {
	    throw new HiLAException(
		    "Can only move files within the same storages");
	}
    }

    /**
     * @see eu.unicore.hila.grid.File#transfer(eu.unicore.hila.grid.File,
     *      boolean)
     */
    @Override
    public ThirdPartyTransfer transfer(File _other, boolean _overwrite,
	    boolean _recursive) throws HiLAException {
	if (_other instanceof Unicore6File) {
	    if (_other.exists() && _other.isDirectory()) {
		_other = (File) _other.getChild(this.getName());
	    }

	    if (_other.exists() && !_overwrite) {
		throw new HiLAException("Won't overwrite existing file: "
			+ _other.getLocation());
	    }

	    if (this.isDirectory()) {
		if (!_recursive) {
		    throw new HiLAException(
			    "Directories can only be copied recursively.");
		} else {
               ThirdPartyTransfer tpt = new Unicore6RecursiveTransferTask(this,
                     (Unicore6File) _other);
		    return tpt;
		}
	    } else {
            return new Unicore6TransferTask(this, this.storageClient,
                  (Unicore6File) _other, ((Unicore6File) _other).storageClient);
	    }
	} else {
	    throw new HiLAException("Can only transfer files of same type.");
	}

    }

    private void updateGridFile() throws HiLAException {
	final long ct = System.currentTimeMillis();
	if ((gridFile == null) || (ct - gridFileTimeStamp) > 500) {
	    if (storageClient == null) {
		if (storage != null) {
		    storageClient = storage.getStorageClient();
		} else {
		    throw new HiLAException("No access to storage.");
		}
	    }
	    try {
		gridFile = storageClient.listProperties(decodedPath);
		gridFileTimeStamp = System.currentTimeMillis();
	    } catch (BaseFault e) {
		if (e.getMessage().contains(REASON_FILENOTFOUND)) {
		    throw new HiLAException("File not found.", e);
		} else {
		    throw new HiLAException(
			    "Unable to update file properties.", e);
		}
	    } catch (FileNotFoundException e) {
		throw new HiLAException("File does not exist.", e);
	    }
	}
    }

    /**
     * 
     */
    public String getStagingURI() {
	return storageClient.getUrl().concat("#/").concat(getPath());
    }

    /**
     * @see eu.unicore.hila.grid.File#exportToStream(java.io.OutputStream)
     */
    @Override
    public SimpleTransfer exportToStream(OutputStream _os) throws HiLAException {
	Location storageLocation = findParentLocationOfType(Storage.class,
		this.location, Grid.class);
	return new Unicore6ExportToStreamTask(storageLocation, storageClient,
		decodedPath, this, _os);
    }

    /**
     * @see eu.unicore.hila.grid.File#importFromStream(java.io.InputStream,
     *      boolean)
     */
    @Override
    public SimpleTransfer importFromStream(InputStream is, boolean overwrite)
	    throws HiLAException {
	if (this.exists() && !overwrite) {
	    throw new HiLAException(
		    "Existing remote file will not be overwritten.");
	}

	SimpleTransfer trsf = new Unicore6ImportFromStreamTask(location,
		storageClient, decodedPath, is);

	return trsf;
    }

}
