/*********************************************************************************
 * Copyright (c) 2010-2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.apache.log4j.Logger;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.StorageManagement;
import eu.unicore.hila.HiLA;
import eu.unicore.hila.Location;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.grid.Config;
import eu.unicore.hila.grid.Storage;

/**
 * @author bjoernh
 * 
 *         27.05.2010 17:02:33
 * 
 */
public class StorageLocator {
    private static final Logger log = Logger.getLogger(StorageLocator.class);

    private static volatile StorageLocator theInstance = null;
    private static Object theInstanceLock = new Object();

    private Config config;

    private Object configLock = new Object();

    /**
     * 
     */
    protected StorageLocator() {

    }

    public static StorageLocator getInstance() {
	if (theInstance == null) {
	    synchronized (theInstanceLock) {
		if (theInstance == null) {
		    theInstance = new StorageLocator();
		}
	    }
	}
	return theInstance;
    }

    @SuppressWarnings("unchecked")
    public List<Storage> getAllStorages(List<String> _registryURLs,
	    Unicore6Properties _props, long timeoutMillis,
	    Location _parentLocation) throws HiLAException {

	if (config == null) {
	    synchronized (configLock) {
		if (config == null) {
		    String configProviderClass = _props
			    .getConfigImplementation();
		    Class<Config> configClass;
		    try {
			configClass = (Class<Config>) Class
				.forName(configProviderClass);
		    } catch (ClassNotFoundException e) {
			throw new HiLAException(
				"hila.unicore6.config references unknown class"
					+ configProviderClass, e);
		    } catch (ClassCastException e) {
			throw new HiLAException(configProviderClass
				+ " is not a subclass of "
				+ Config.class.getName());
		    }
		    Constructor<Config> constructor;
		    try {
			constructor = configClass
				.getConstructor(Unicore6Properties.class);
		    } catch (SecurityException e) {
			throw new HiLAException(e.getMessage(), e);
		    } catch (NoSuchMethodException e) {
			throw new HiLAException(
				"Config implementation referenced in "
					+ "hila.unicore6.config does not provide the "
					+ "correct constructor", e);
		    }
		    try {
			config = constructor.newInstance(_props);
		    } catch (IllegalArgumentException e) {
			throw new HiLAException(e.getMessage(), e);
		    } catch (InstantiationException e) {
			throw new HiLAException(e.getMessage(), e);
		    } catch (IllegalAccessException e) {
			throw new HiLAException(e.getMessage(), e);
		    } catch (InvocationTargetException e) {
			throw new HiLAException(e.getMessage(), e);
		    }
		}
	    }

	}

	Unicore6SecurityProperties secProps = (Unicore6SecurityProperties) config
		.getIdForLocation(_parentLocation);

	if (secProps != null) {
	    secProps = (Unicore6SecurityProperties) secProps.clone();
	}

	List<Storage> storages = new ArrayList<Storage>();
	List<Future<List<EndpointReferenceType>>> storageEPRFutures = new ArrayList<Future<List<EndpointReferenceType>>>();
	for (String registryURL : _registryURLs) {
	    GetServicesFromRegistry getSMSs = new GetServicesFromRegistry(
		    registryURL, secProps, StorageManagement.SMS_PORT);
         storageEPRFutures.add(HiLA.getExecutor().submit(getSMSs));
	}

	final long registryDeadline = System.currentTimeMillis()
		+ timeoutMillis;

	for (Future<List<EndpointReferenceType>> storageFuture : storageEPRFutures) {
	    try {
		List<EndpointReferenceType> storageEPRs = storageFuture.get(
			registryDeadline - System.currentTimeMillis(),
			TimeUnit.MILLISECONDS);
		List<Future<Storage>> storageFutures = new ArrayList<Future<Storage>>();
		for (EndpointReferenceType sEPR : storageEPRs) {
		    GetStorageFromEPR getStorageFromEPR = new GetStorageFromEPR(
			    sEPR, secProps, _parentLocation);
               storageFutures.add(HiLA.getExecutor().submit(getStorageFromEPR));
		}

		final long storageDeadline = System.currentTimeMillis()
			+ timeoutMillis;

		for (Future<Storage> future : storageFutures) {
		    try {
			storages.add(future.get(
				storageDeadline - System.currentTimeMillis(),
				TimeUnit.MILLISECONDS));
		    } catch (InterruptedException e) {
			log.error(
				"Interrupted while waiting for storage service.",
				e);
		    } catch (ExecutionException e) {
			log.error("Error retrieving storage.", e);
		    } catch (TimeoutException e) {
			log.error("Timeout while waiting for storage service.",
				e);
		    }
		}
	    } catch (InterruptedException e) {
		log.error("Interrupted while waiting for registry.", e);
	    } catch (ExecutionException e) {
		log.error("Error retrieving storage references.", e);
	    } catch (TimeoutException e) {
		log.error("Timeout waiting for registry.", e);
	    }
	}

	return storages;
    }
}
