/*********************************************************************************
 * Copyright (c) 2009-2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;

import org.apache.log4j.Logger;
import org.codehaus.xfire.fault.XFireFault;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.client.TSFClient;
import de.fzj.unicore.uas.client.TSSClient;
import eu.unicore.hila.Location;
import eu.unicore.hila.exceptions.HiLAAccessDeniedException;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.grid.Site;
import eu.unicore.util.httpclient.IClientConfiguration;

/**
 * An internal helper class of the UNICORE 6 implementation. It is designed as a
 * callable in order to allow parallel discovery of multiple sites. Also,
 * timeouts may be used in order to cover for not responding sites or dropped
 * network connections.
 * 
 * @author bjoernh
 * 
 *         30.09.2009 14:54:29
 * 
 */
class GetSitesFromTSF implements Callable<List<Site>> {

    private static final Logger log = Logger.getLogger(GetSitesFromTSF.class);

    private EndpointReferenceType tsfEpr = null;
    private IClientConfiguration security;
    private Location location;
    private boolean separateTSSs = false;

    /**
     * @param _location
     * 
     */
    public GetSitesFromTSF(EndpointReferenceType _tsfURL,
            IClientConfiguration secProps, Location _location,
	    boolean _separateTSSs) {
	if (log.isDebugEnabled()) {
	    log.debug("Initializing " + this + " with _tsfURL: " + _tsfURL
		    + ", _location: " + _location + ", _separateTSSs: "
		    + _separateTSSs);
	}
	this.tsfEpr = _tsfURL;
	this.security = secProps;
	this.location = _location;
	this.separateTSSs = _separateTSSs;
    }

    /**
     * @see java.util.concurrent.Callable#call()
     */
    public List<Site> call() throws Exception {
	final List<Site> sites = new ArrayList<Site>();
	final TSFClient tsfClient = new TSFClient(tsfEpr.getAddress()
		.getStringValue(), tsfEpr, security);

	try {
	    final List<EndpointReferenceType> existingTssEprs = tsfClient
		    .getAccessibleTargetSystems();
	    if (log.isDebugEnabled()) {
		log.debug("Found " + existingTssEprs.size()
			+ " Target Systems entries in TSF.");
		if (log.isTraceEnabled()) {
		    for (EndpointReferenceType endpointReference : existingTssEprs) {
			log.trace(endpointReference);
		    }
		}
	    }
	    final List<EndpointReferenceType> newTssEprs = new ArrayList<EndpointReferenceType>();
	    if ((existingTssEprs == null) || (existingTssEprs.size() == 0)) {
		final TSSClient tssClient = tsfClient.createTSS();
		if (tssClient != null) {
		    newTssEprs.add(tssClient.getEPR());
		}
	    }
	    if (existingTssEprs != null) {
		newTssEprs.addAll(existingTssEprs);
	    }
	    if (separateTSSs) {
		if (log.isDebugEnabled()) {
		    log.debug("Generating separate Sites for TSSs.");
		}
		for (EndpointReferenceType tssEpr : newTssEprs) {
		    final TSSClient tssClient = new TSSClient(tssEpr
			    .getAddress().getStringValue(), tssEpr,
			    this.security);
		    try {
			final String siteName = tssClient.getTargetSystemName();
			final Calendar upSince = tssClient.getUpSince();
			final String dateString = constructDateString(upSince);
			final Location siteLocation = location
				.getChildLocation(siteName.concat("_").concat(
					dateString));
			if (log.isTraceEnabled()) {
			    log.trace("Generating site " + siteLocation);
			}
                  sites.add((Unicore6Site) siteLocation.locate(
                        Collections.singletonList(tssClient), tsfClient));
		    } catch (Exception e) {
			// thrown if TSS not accessible (for whatever
			// reason)
                  log.error("Something went wrong.", e);
		    }
		}
	    } else {
		if (log.isDebugEnabled()) {
		    log.debug("Generating single site for multiple TSSs.");
		}
		final List<TSSClient> tssClients = new ArrayList<TSSClient>();
		for (EndpointReferenceType tssEpr : newTssEprs) {
		    final TSSClient tssClient = new TSSClient(tssEpr
			    .getAddress().getStringValue(), tssEpr,
			    this.security);
		    if (log.isTraceEnabled()) {
			log.trace("Adding " + tssClient + " to list.");
		    }
		    tssClients.add(tssClient);
		}
		sites.add((Unicore6Site) location.getChildLocation(
                  tssClients.get(0).getTargetSystemName()).locate(tssClients,
                  tsfClient));
	    }
	} catch (XFireFault e) {
	    if (e.getMessage().contains("Access denied")) {
		throw new HiLAAccessDeniedException(e.getMessage());
	    }
	    throw new HiLAException(e.getMessage());
	} catch (Exception e) {
	    // thrown if TSF not accessible (for whatever reason)
	    throw new HiLAException(e.getMessage());
	}
	if(log.isDebugEnabled()) {
	    log.debug("Returning list of " + sites.size() + " sites.");
	}
	return sites;
    }

    /**
     * @param upSince
     * @return
     */
    public static String constructDateString(final Calendar upSince) {
	final String dateString = Integer.toString(upSince.get(Calendar.YEAR))
		+ (upSince.get(Calendar.MONTH) + 1 < 10 ? "0" : "")
		+ Integer.toString(upSince.get(Calendar.MONTH) + 1)
		+ (upSince.get(Calendar.DAY_OF_MONTH) < 10 ? "0" : "")
		+ Integer.toString(upSince.get(Calendar.DAY_OF_MONTH))
		+ (upSince.get(Calendar.HOUR_OF_DAY) < 10 ? "0" : "")
		+ Integer.toString(upSince.get(Calendar.HOUR_OF_DAY))
		+ (upSince.get(Calendar.MINUTE) < 10 ? "0" : "")
		+ Integer.toString(upSince.get(Calendar.MINUTE));
	return dateString;
    }
}
