/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.shell.commands;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.ParseException;
import org.apache.log4j.Logger;

import eu.unicore.hila.Location;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.grid.Job;
import eu.unicore.hila.grid.TaskStatus;
import eu.unicore.hila.shell.ShellContext;

/**
 * @author bjoernh
 * 
 *         16.10.2009 15:53:48
 * 
 */
public class Start extends ShellCommandBase implements ShellCommand {
    private static final Logger log = Logger.getLogger(Start.class);

    /**
     * 
     */
    public Start() {
	initializeOptions();
    }

    /**
     * @see eu.unicore.hila.shell.commands.ShellCommandBase#execute(java.lang.String[],
     *      eu.unicore.hila.shell.ShellContext)
     */
    @Override
    public void execute(String[] cmdArgs, ShellContext ctx)
	    throws HiLAException {
	try {
	    CommandLine cmdLine = parser.parse(options, cmdArgs);
	    if (cmdLine.hasOption("h")) {
		printHelp(ctx);
		return;
	    }

	    List<String> arguments = cmdLine.getArgList();
	    arguments.remove(0); // command name itself is not an argument
	    final List<Job> tasks = new ArrayList<Job>();
	    for (String taskLocation : arguments) {
		Location location = new Location(taskLocation);
		if(!location.isLocationOfType(Job.class)) {
		    // resolve relative
		    location = ctx.getCurrentLocation().getChildLocation(taskLocation);
		}
		if (location.isLocationOfType(Job.class)) {
		    Job task = (Job) location.locate();
		    if (task != null) {
			tasks.add(task);
		    } else {
			log.warn(taskLocation + " does not exist.");
		    }
		} else {
		    log.warn(taskLocation + " is not a valid Task location.");
		}
	    }

	    for (Job task : tasks) {
		task.startASync();
	    }

	    if (cmdLine.hasOption("b")) {
		String blockMS = cmdLine.getOptionValue("b");

		for (Job computeTask : tasks) {
		    TaskStatus finalStatus = null;
		    if (blockMS == null) {
			finalStatus = computeTask.block();
		    } else {
			long block = Long.parseLong(blockMS);
			finalStatus = computeTask.block(block);
		    }
		    try {
			ctx.getCs().printString(
				computeTask.getLocation().toString() + " "
					+ finalStatus.toString());
			ctx.getCs().printNewline();
		    } catch (IOException e) {
			log.info(computeTask.getLocation().toString() + " "
				+ finalStatus.toString());
		    }
		}
	    }

	} catch (ParseException e) {
	    log.error("Unable to parse command line.", e);
	}

    }

    /**
     * 
     */
    private void initializeOptions() {
	options.addOption(OptionBuilder.withArgName("h").withDescription(
		"Print this help message.").withLongOpt("help").create("h"));
	options
		.addOption(OptionBuilder
			.withArgName("b")
			.withDescription(
				"Block on task. Optional argument for blocking limited number of ms.")
			.hasOptionalArgs(1).withArgName("block").create("b"));
    }
}
