/*********************************************************************************
 * Copyright (c) 2010 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.shell.commands;

import java.io.IOException;
import java.io.InputStream;
import java.util.Calendar;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.ParseException;
import org.apache.log4j.Logger;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.RangeValueType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ResourcesType;

import eu.unicore.hila.Location;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.grid.File;
import eu.unicore.hila.grid.Job;
import eu.unicore.hila.grid.SimpleTransfer;
import eu.unicore.hila.grid.Site;
import eu.unicore.hila.grid.TaskStatus;
import eu.unicore.hila.job.model.JobModel;
import eu.unicore.hila.shell.ShellContext;

/**
 * @author bjoernh
 * 
 *         12.05.2010 10:33:01
 * 
 */
public class Run extends ShellCommandBase implements ShellCommand {
	private static final Logger log = Logger.getLogger(Run.class);

	/**
     * 
     */
	public Run() {
		initializeOptions();
	}

	/**
     * 
     */
	private void initializeOptions() {
		options.addOption(OptionConstants.HELP);
		options.addOption(OptionConstants.APP_NAME);
		options.addOption(OptionConstants.APP_VERSION);
		options.addOption(OptionConstants.APP_EXECUTABLE);
		options.addOption(OptionConstants.APP_ARGUMENT);
		options.addOption(OptionConstants.SITE);
		options.addOption(OptionBuilder.withArgName("wtime")
				.withDescription("Wall clock time.").hasArg(true).create('w'));
	}

	/**
	 * @see eu.unicore.hila.shell.commands.ShellCommandBase#execute(java.lang.String[],
	 *      eu.unicore.hila.shell.ShellContext)
	 */
	@Override
	public void execute(String[] cmdArgs, ShellContext ctx)
			throws HiLAException {
		InputStream is = null;
		try {
			CommandLine cl = parser.parse(options, cmdArgs);

			if (cl.hasOption(OptionConstants.HELP.getArgName())) {
				printHelp(ctx);
				return;
			}

			if (log.isDebugEnabled()) {
				log.debug("Locating site for Execution.");
			}
			Site executionSite = null;
			if (cl.hasOption(OptionConstants.SITE.getOpt())) {
				final String siteLocStr = cl
						.getOptionValue(OptionConstants.SITE.getOpt());
				Location siteLoc = new Location(siteLocStr);
				if (!siteLoc.isLocationOfType(Site.class)) {
					siteLoc = ctx.getCurrentLocation().getChildLocation(
							siteLocStr);
				}
				if (!siteLoc.isLocationOfType(Site.class)) {
					log.error("Site location invalid.");
					return;
				}
				executionSite = (Site) siteLoc.locate();
			} else {
				log.error("No site given for execution.");
				return;
			}

			if (log.isDebugEnabled()) {
				log.debug("Creating job model.");
			}
			final JobModel jm = new JobModel();

			// executable
			if (cl.hasOption(OptionConstants.APP_EXECUTABLE.getOpt())) {
				String aeStr = cl.getOptionValue(OptionConstants.APP_EXECUTABLE
						.getOpt());
				jm.setExecutable(aeStr);
			}

			if (cl.hasOption(OptionConstants.APP_ARGUMENT.getOpt())) {
				String[] arguments = cl
						.getOptionValues(OptionConstants.APP_ARGUMENT.getOpt());

				for (String argument : arguments) {
					jm.addArgument(argument);
				}
			}

			// resources
			ResourcesType rt = ResourcesType.Factory.newInstance();
			if (cl.hasOption('w')) {
				String sWtime = cl.getOptionValue('w');
				int wtime = Integer.parseInt(sWtime);

				RangeValueType rv = rt.addNewIndividualCPUTime();
				rv.addNewExact().setDoubleValue(wtime);
			}
			jm.setResources(rt);
			if (jm.getTaskName() == null) {
				jm.setTaskName("HiLA Shell 'run' job submitted at "
						+ Calendar.getInstance().getTime());
			}

			if (log.isDebugEnabled()) {
				log.debug("Submitting job.");
			}

			jm.setStdout("stdout");
			jm.setStderr("stderr");

			Job task = executionSite.submit(jm);
			if (log.isDebugEnabled()) {
				log.debug("Job submitted as " + task.getLocation().toString());
			}
			task.startSync();

			if (log.isDebugEnabled()) {
				log.debug("Waiting for job to finish ...");
			}
			while (!isFinalState(task.status())) {
				try {
					Thread.sleep(1000);
				} catch (InterruptedException e) {

				}
			}

			File stdout = task.getStdOut();

			if (log.isDebugEnabled()) {
				log.debug("Retrieving stdout from " + stdout.getLocation());
			}
			SimpleTransfer transfer = stdout.exportToStream(System.out);
			transfer.block();
		} catch (HiLAException e) {
			log.error(e.getMessage(), e);
		} catch (ParseException e) {
			log.error("Unable to parse command line.", e);
		} finally {
			try {
				if (is != null) {
					is.close();
				}
			} catch (IOException e) {
				log.error("Unable to close stream.", e);
			}
		}
	}

	/**
	 * @param status
	 * @return
	 */
	private boolean isFinalState(TaskStatus status) {
		return status.equals(TaskStatus.ABORTED)
				|| status.equals(TaskStatus.FAILED)
				|| status.equals(TaskStatus.SUCCESSFUL);
	}
}
