/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.shell.commands;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.ParseException;
import org.apache.log4j.Logger;

import eu.unicore.hila.Location;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLALocationSyntaxException;
import eu.unicore.hila.grid.SimpleTransfer;
import eu.unicore.hila.shell.ShellContext;

/**
 * @author bjoernh
 * 
 *         16.10.2009 15:54:50
 * 
 */
public class Export extends ShellCommandBase implements ShellCommand {
    private static final Logger log = Logger.getLogger(Export.class);

    /**
     * 
     */
    public Export() {
	super();
	initializeOptions();
    }

    /**
     * 
     */
    private void initializeOptions() {
	options.addOption(OptionConstants.BLOCK);
	options.addOption(OptionConstants.HELP);
	options.addOption(OptionConstants.FORCE);
	options.addOption(OptionConstants.RECURSIVE);
    }

    /**
     * @see eu.unicore.hila.shell.commands.ShellCommand#execute(java.lang.String[],
     *      eu.unicore.hila.shell.ShellContext)
     */
    public void execute(String[] cmdArgs, ShellContext ctx) {
	CommandLine cmdline = null;
	try {
	    cmdline = parser.parse(options, cmdArgs);
	    if (cmdline.hasOption(OptionConstants.HELP.getOpt())) {
		printHelp(ctx);
		return;
	    }
	} catch (ParseException e) {
	    log.error("Unable to parse command line", e);
	    return;
	}

	List<String> args = cmdline.getArgList();
	args.remove(0); // remove the command itself
	if (args.size() == 0) {
	    log.error("Nothing to export.");
	    return;
	}

	String localPath = args.get(args.size() - 1);
	File localFile;
	if (args.size() == 1) {
	    localFile = ctx.getCurrentLocalDir();
	} else {
	    localFile = new File(localPath);
	}

	if (!localFile.isAbsolute()) {
	    localFile = new File(ctx.getCurrentLocalDir(), localPath);
	}

	if (localFile.exists()) {
	    if (!localFile.isDirectory()) {
		if (args.size() > 2) {
		    log.error(localFile.toString() + " is not a directory.");
		    return;
		} else {
		    if (!cmdline.hasOption(OptionConstants.FORCE.getOpt())) {
			log.error("Use " + OptionConstants.FORCE.getOpt()
				+ " to overwrite existing file.");
			return;
		    }
		}
	    }
	} else { // localFile does not exist
	    if (args.size() > 2) {
		// cannot copy to non-existing directory
		log.error("Cannot copy multiple files to non-existent directory "
			+ localPath);
		return;
	    } else if (localFile.getParentFile().exists()
		    && !localFile.getParentFile().isDirectory()) {
		log.error("Cannot create file in given location.");
		return;
	    }
	}

	List<SimpleTransfer> transfers = new ArrayList<SimpleTransfer>();
	// all arguments except the last are files to be exported
	for (int i = 0; i < args.size() - 1; ++i) {
	    try {
		Location loc = new Location(args.get(i));
		// try resolving relative remote path
		if (!loc.isLocationOfType(eu.unicore.hila.grid.File.class)) {
		    loc = ctx.getCurrentLocation()
			    .getChildLocation(args.get(i));
		}
		if (loc.isLocationOfType(eu.unicore.hila.grid.File.class)) {
		    try {
			transfers
				.add(((eu.unicore.hila.grid.File) loc.locate()).exportToLocalFile(
					localFile,
					cmdline.hasOption(OptionConstants.FORCE
						.getArgName()),
					cmdline.hasOption(OptionConstants.RECURSIVE
						.getArgName())));
		    } catch (HiLAException e) {
			log.error("Export of " + loc.toString() + " failed.", e);
		    }
		} else {
		    log.warn(loc.toString() + " not of type File.");
		}
	    } catch (HiLALocationSyntaxException e) {
		log.error(
			"Invalid location syntax, continuing with next location.",
			e);
	    }
	}
	for (SimpleTransfer simpleTransfer : transfers) {
	    log.info(simpleTransfer.getLocation());
	}
    }

    /**
     * @see eu.unicore.hila.shell.commands.ShellCommand#help()
     */
    public String help() {
	StringBuilder sb = new StringBuilder(
		"Export file from a storage to the local file system.");
	sb.append(System.getProperty("line.separator"));
	sb.append("export <Source File Location> <Local File Path>");
	sb.append(System.getProperty("line.separator"));
	return sb.append(super.help()).toString();
    }

}
