/*********************************************************************************
 * Copyright (c) 2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.job.jsdl;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.CreationFlagEnumeration.Enum;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.DataStagingType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDefinitionDocument;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDescriptionType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobIdentificationType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ResourcesType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.SourceTargetType;
import org.ggf.schemas.jsdl.x2005.x11.jsdlPosix.ArgumentType;
import org.ggf.schemas.jsdl.x2005.x11.jsdlPosix.EnvironmentType;
import org.ggf.schemas.jsdl.x2005.x11.jsdlPosix.POSIXApplicationDocument;
import org.ggf.schemas.jsdl.x2005.x11.jsdlPosix.POSIXApplicationType;

import eu.unicore.hila.job.model.JobModel;
import eu.unicore.hila.job.model.StageIn;
import eu.unicore.hila.job.model.StageOut;
import eu.unicore.hila.job.spi.NativeToModel;

/**
 * @author bjoernh
 * 
 *         22.06.2011 16:33:23
 * 
 */
public class JSDLtoModel implements NativeToModel {
   private final static Logger log = Logger.getLogger(JSDLtoModel.class);
   private JobDefinitionDocument jsdl = null;
   private JobModel model;

   /**
       * 
       */
   public JSDLtoModel(Object _jsdl) {
      this.jsdl = (JobDefinitionDocument) _jsdl;
   }

   public JobModel getModel() {
      if (this.model == null) {
         transform();
      }
      return this.model;
   }

   private void transform() {
      this.model = new JobModel();

      JobDescriptionType jDescr = jsdl.getJobDefinition().getJobDescription();

      POSIXApplicationType posixApp = null;
      if (jDescr.getApplication().selectChildren(
            POSIXApplicationType.type.getDocumentElementName()).length > 0) {
         XmlObject[] posixNodes = jDescr.getApplication().selectChildren(
               POSIXApplicationDocument.type.getDocumentElementName());
         try {
            if (posixNodes.length > 0) {
               posixApp = POSIXApplicationType.Factory.parse(posixNodes[0]
                     .xmlText());
            }
            // posixApp = POSIXApplicationDocument.Factory.parse(jDescr.);
         } catch (XmlException e) {
            log.error("Unable to parse POSIXApplication element", e);
         }
      }

      // application or executable
      if (jDescr.isSetApplication()) {
         if (jDescr.getApplication().isSetApplicationName()) {
            model.setApplicationName(jDescr.getApplication()
                  .getApplicationName());
            if (jDescr.getApplication().isSetApplicationVersion()) {
               model.setApplicationVersion(jDescr.getApplication()
                     .getApplicationVersion());
            }
         } else if (posixApp != null) {
            model.setExecutable(posixApp.getExecutable().getStringValue());
         }

         if (jDescr.getApplication().isSetDescription()) {
            model.setApplicationDescription(jDescr.getApplication()
                  .getDescription());
         }
      }

      if (posixApp != null) {
         ArgumentType[] arguments = posixApp.getArgumentArray();
         for (ArgumentType argument : arguments) {
            model.addArgument(argument.getStringValue());
         }

         EnvironmentType[] environment = posixApp.getEnvironmentArray();
         for (EnvironmentType environmentEntry : environment) {
            model.addEnvironment(environmentEntry.getName(),
                  environmentEntry.getStringValue());
         }

			if (posixApp.isSetInput()) {
				model.setStdin(posixApp.getInput().getStringValue());
			}
			if (posixApp.isSetOutput()) {
				model.setStdout(posixApp.getOutput().getStringValue());
			}
			if (posixApp.isSetError()) {
				model.setStderr(posixApp.getError().getStringValue());
			}
      }

      // job identity
      if (jDescr.isSetJobIdentification()) {
         final JobIdentificationType jobId = jDescr.getJobIdentification();
         // job name
         if (jobId.isSetJobName()) {
            model.setTaskName(jobId.getJobName());
         }
         // job annotation
         final String[] annotations = jobId.getJobAnnotationArray();
         if (annotations != null) {
            for (String annotation : annotations) {
               model.addJobAnnotation(annotation);
            }
         }
         // job description
         if (jobId.isSetDescription()) {
            model.setJobDescription(jobId.getDescription());
         }
         // job project
         final String[] projects = jobId.getJobProjectArray();
         if (projects != null) {
            for (String project : projects) {
               model.addJobProject(project);
            }
         }

      }

      // copy resources from JSDL
		if (jDescr.isSetResources()) {
			model.setResources((ResourcesType) jDescr.getResources().copy());
		}

      // data staging
      final DataStagingType[] dataStagings = jDescr.getDataStagingArray();
      if (dataStagings != null) {
         for (DataStagingType staging : dataStagings) {
            final String fileName = staging.getFileName();
            String fileSystemName = null;
            if (staging.isSetFilesystemName()) {
               fileSystemName = staging.getFilesystemName();
            }
            // TODO: validate existence of corresponding filesystem entry
            Enum creationFlag = staging.getCreationFlag();
            boolean deleteOnTermination = staging.getDeleteOnTermination();
            // TODO: What's the meaning of URI element unset?
            String sourceURI = null;
            if (staging.isSetSource()) {
               SourceTargetType source = staging.getSource();
               if (source.isSetURI()) {
                  sourceURI = source.getURI();
               }
            }
            String targetURI = null;
            if (staging.isSetTarget()) {
               SourceTargetType target = staging.getTarget();
               if (target.isSetURI()) {
                  targetURI = target.getURI();
               }
            }

            if (targetURI != null) {
               StageOut stageout = new StageOut(fileName, targetURI,
                     fileSystemName, creationFlag, deleteOnTermination);
               model.addStageOut(stageout);
            }
            if (sourceURI != null) {
               StageIn stageIn = new StageIn(fileName, sourceURI,
                     fileSystemName, creationFlag, deleteOnTermination);
               model.addStageIn(stageIn);
            }
         }
      }

   }

}
