/*********************************************************************************
 * Copyright (c) 2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.job.emi;

import java.math.BigInteger;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlOptions;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.CPUArchitectureType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.OperatingSystemType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.OperatingSystemTypeEnumeration;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ProcessorArchitectureEnumeration;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ResourcesType;

import eu.emi.es.x2010.x12.adl.ActivityDescriptionDocument;
import eu.emi.es.x2010.x12.adl.ActivityDescriptionDocument.ActivityDescription;
import eu.emi.es.x2010.x12.adl.ActivityIdentificationDocument.ActivityIdentification;
import eu.emi.es.x2010.x12.adl.ApplicationDocument.Application;
import eu.emi.es.x2010.x12.adl.DataStagingDocument.DataStaging;
import eu.emi.es.x2010.x12.adl.ExecutableType;
import eu.emi.es.x2010.x12.adl.InputFileDocument.InputFile;
import eu.emi.es.x2010.x12.adl.OperatingSystemDocument.OperatingSystem;
import eu.emi.es.x2010.x12.adl.OptionType;
import eu.emi.es.x2010.x12.adl.OutputFileDocument.OutputFile;
import eu.emi.es.x2010.x12.adl.ResourcesDocument.Resources;
import eu.emi.es.x2010.x12.adl.RuntimeEnvironmentDocument.RuntimeEnvironment;
import eu.emi.es.x2010.x12.adl.SourceDocument.Source;
import eu.emi.es.x2010.x12.adl.TargetDocument.Target;
import eu.unicore.hila.job.model.JobModel;
import eu.unicore.hila.job.model.StageIn;
import eu.unicore.hila.job.model.StageOut;
import eu.unicore.hila.job.spi.ModelToNative;

/**
 * @author bjoernh
 * 
 *         21.12.2011 12:43:19
 * 
 */
public class ModelToEmiAdl implements ModelToNative {

	private static final Logger log = Logger.getLogger(ModelToEmiAdl.class);

	/**
    * 
    */
	protected static final String EMI_ADL = "emi-adl";
	private JobModel model;
	private long lastAdlUpdate = 0;
	private ActivityDescriptionDocument adl;

	/**
	 * @param _model
	 *            the model to set
	 */
	public ModelToEmiAdl(JobModel _model) {
		this.model = _model;
	}

	/**
	 * @see eu.unicore.hila.job.spi.ModelToNative#getNative()
	 */
	@Override
	public Object getNative() {
		if (lastAdlUpdate < model.getLastModified()) {
			createAdl();
		}
		if (log.isDebugEnabled()) {
			log.debug("EMI ADL:" + System.getProperty("line.separator")
					+ this.adl.xmlText(new XmlOptions().setSavePrettyPrint()));
		}
		return this.adl;
	}

	/**
    * 
    */
	private void createAdl() {
		adl = ActivityDescriptionDocument.Factory.newInstance();
		ActivityDescription ad = adl.addNewActivityDescription();

		// resources
		if (model.getResources() != null) {
			Resources res = transformResources();
			ad.setResources(res);
		}

		// applicationName
		if (model.getApplicationName() != null) {
			if (ad.getResources() == null) {
				ad.addNewResources();
			}
			RuntimeEnvironment rte = ad.getResources()
					.addNewRuntimeEnvironment();
			rte.setName(model.getApplicationName());
			rte.setVersion(model.getApplicationVersion());
		}

		// executable
		Application application = null;
		ExecutableType executable = null;
		if (model.getExecutable() != null) {
			application = ad.addNewApplication();
			executable = application.addNewExecutable();
			executable.setPath(model.getExecutable());
		}
		// arguments
		if (model.getArguments() != null && !model.getArguments().isEmpty()) {
			application = ensureApplication(ad);
			executable = ensureExecutable(ad);
			for (String argument : model.getArguments()) {
				executable.addArgument(argument);
			}
		}
		// environment
		if (model.getEnvironment() != null && !model.getEnvironment().isEmpty()) {
			executable = ensureExecutable(ad);
			for (String envName : model.getEnvironment().keySet()) {
				OptionType option = application.addNewEnvironment();
				option.setName(envName);
				option.setValue(model.getEnvironment().get(envName));
			}
		}

		// exports and imports need to be dealt with by the implementation, as
		// they are intended to be local imports/exports, i.e. require direct
		// interaction between client and server
		// exports
		// imports

		// jobProjects

		// stageIns
		// TODO this is only a very basic implementation, not respecting all
		// features of EMI-ADL nor the abstract model
		DataStaging dataStaging = null;
		if (model.getStageIns() != null && !model.getStageIns().isEmpty()) {
			dataStaging = ensureDataStaging(ad);
			for (StageIn stageIn : model.getStageIns()) {
				InputFile inputFile = dataStaging.addNewInputFile();
				Source source = inputFile.addNewSource();
				source.setURI(stageIn.getSourceURI());
				inputFile.setName(stageIn.getFileName());
			}
		}
		// stageOuts
		// TODO this is only a very basic implementation, not respecting all
		// features of EMI-ADL nor the abstract model
		if (model.getStageOuts() != null && !model.getStageOuts().isEmpty()) {
			dataStaging = ensureDataStaging(ad);
			for (StageOut stageOut : model.getStageOuts()) {
				OutputFile outputFile = dataStaging.addNewOutputFile();
				Target target = outputFile.addNewTarget();
				target.setURI(stageOut.getTargetURI());
				outputFile.setName(stageOut.getFileName());
			}
		}

		ActivityIdentification activityId = null;
		// taskName
		if (model.getTaskName() != null) {
			activityId = ensureActivityIdentification(ad);
			activityId.setName(model.getTaskName());
		}
		// applicationDescription
		if (model.getApplicationDescription() != null) {
			activityId = ensureActivityIdentification(ad);
			activityId.setDescription(model.getApplicationDescription());
		}
		// jobAnnotations
		// TODO classpath resolution problems

		// jobDescription
		if (model.getJobDescription() != null) {
			activityId = ensureActivityIdentification(ad);
			activityId.setDescription(model.getJobDescription());
		}

		// What if the DataStaging already exists? This is a bit skewed. Going
		// back and forth between model and native will lead to an increased
		// number of DataStagings.
		// This is required (at least) by the Arc implementation, otherwise
		// files cannot be retrieved at all
		if (model.getStdout() != null) {
			ad.getApplication().setOutput(model.getStdout());
			if (!ad.isSetDataStaging()) {
				ad.addNewDataStaging();
			}
			ad.getDataStaging().addNewOutputFile().setName(model.getStdout());
		}
		if (model.getStderr() != null) {
			ad.getApplication().setError(model.getStderr());
			if (!ad.isSetDataStaging()) {
				ad.addNewDataStaging();
			}
			ad.getDataStaging().addNewOutputFile().setName(model.getStderr());
		}
		if (model.getStdin() != null) {
			ad.getApplication().setInput(model.getStdout());
		}

		lastAdlUpdate = System.currentTimeMillis();
	}

	/**
	 * @return
	 */
	private Resources transformResources() {
		ResourcesType jsdlRes = model.getResources();
		Resources res = Resources.Factory.newInstance();

		if (jsdlRes.isSetCandidateHosts()) {
			log.warn("Cannot map CandidateHosts to EMI ADL.");
		}

		if (jsdlRes.isSetCPUArchitecture()) {
			res.setPlatform(mapPlatform(jsdlRes.getCPUArchitecture()));
		}
		if (jsdlRes.isSetExclusiveExecution()) {
			log.warn("Cannot request exclusive execution in EMI ADL.");
		}
		if (jsdlRes.isSetIndividualCPUCount()) {
			// TODO
		}
		if (jsdlRes.isSetIndividualCPUSpeed()) {
			// TODO
		}
		if (jsdlRes.isSetIndividualCPUTime()) {
			res.setIndividualCPUTime(BigInteger.valueOf(new Double(jsdlRes
					.getIndividualCPUTime().getExactArray(0).getDoubleValue())
					.longValue()));
		}
		if (jsdlRes.isSetIndividualDiskSpace()) {
			res.setDiskSpaceRequirement(BigInteger
					.valueOf(new Double(jsdlRes.getIndividualDiskSpace()
							.getExactArray(0).getDoubleValue()).longValue()));
		}
		if (jsdlRes.isSetIndividualNetworkBandwidth()) {
			// TODO
		}

		// OS name/family/version
		if (jsdlRes.isSetOperatingSystem()) {
			res.setOperatingSystemArray(0, mapOS(jsdlRes.getOperatingSystem()));
		}

		if (jsdlRes.isSetIndividualPhysicalMemory()) {
			res.setIndividualPhysicalMemory(BigInteger.valueOf(new Double(
					jsdlRes.getIndividualPhysicalMemory().getExactArray(0)
							.getDoubleValue()).longValue()));
		}

		if (jsdlRes.isSetIndividualVirtualMemory()) {
			res.setIndividualVirtualMemory(BigInteger.valueOf(new Double(
					jsdlRes.getIndividualVirtualMemory().getExactArray(0)
							.getDoubleValue()).longValue()));
		}
		if (jsdlRes.isSetTotalCPUCount()) {
			// TODO
		}

		return res;
	}

	/**
	 * Map JSDL CPU architecture names to the ones suggested by EMI ADL. Names
	 * that cannot be mapped directly, will be copied literally.
	 * 
	 * @param cpuArchitecture
	 * @return
	 */
	private String mapPlatform(CPUArchitectureType cpuArchitecture) {
		switch (cpuArchitecture.getCPUArchitectureName().intValue()) {
		case ProcessorArchitectureEnumeration.INT_X_86_64:
			return "amd64";
		case ProcessorArchitectureEnumeration.INT_IA_64:
			return "itanium";
		case ProcessorArchitectureEnumeration.INT_X_86_32:
		case ProcessorArchitectureEnumeration.INT_X_86:
			return "i386";
		case ProcessorArchitectureEnumeration.INT_POWERPC:
			return "powerpc";
		case ProcessorArchitectureEnumeration.INT_SPARC:
			return "sparc";
		default:
			return cpuArchitecture.getCPUArchitectureName().toString();
		}
	}

	/**
	 * Map JSDL Operating system to corresponding names and families suggested
	 * by EMI ADL. Default is to copy name and version.
	 * 
	 * @param operatingSystem
	 * @return
	 */
	private OperatingSystem mapOS(OperatingSystemType operatingSystem) {
		OperatingSystem os = OperatingSystem.Factory.newInstance();
		switch (operatingSystem.getOperatingSystemType()
				.getOperatingSystemName().intValue()) {
		case OperatingSystemTypeEnumeration.INT_LINUX:
			os.setName(OperatingSystemTypeEnumeration.LINUX.toString());
			os.setFamily("linux");
			os.setVersion(operatingSystem.getOperatingSystemVersion());
			break;
		case OperatingSystemTypeEnumeration.INT_MACOS:
			os.setName(OperatingSystemTypeEnumeration.MACOS.toString());
			os.setFamily("macosx");
			os.setVersion(operatingSystem.getOperatingSystemVersion());
			break;
		case OperatingSystemTypeEnumeration.INT_SOLARIS:
			os.setName(OperatingSystemTypeEnumeration.SOLARIS.toString());
			os.setFamily("solaris");
			os.setVersion(operatingSystem.getOperatingSystemVersion());
			break;
		case OperatingSystemTypeEnumeration.INT_WIN_3_X:
		case OperatingSystemTypeEnumeration.INT_WIN_95:
		case OperatingSystemTypeEnumeration.INT_WIN_98:
		case OperatingSystemTypeEnumeration.INT_WINCE:
		case OperatingSystemTypeEnumeration.INT_WINDOWS_2000:
		case OperatingSystemTypeEnumeration.INT_WINDOWS_R_ME:
		case OperatingSystemTypeEnumeration.INT_WINDOWS_XP:
		case OperatingSystemTypeEnumeration.INT_WINNT:
			os.setName(operatingSystem.getOperatingSystemType()
					.getOperatingSystemName().toString());
			os.setVersion(operatingSystem.getOperatingSystemVersion());
			os.setFamily("windows");
			// all others can only be mapped to name/version
		default:
			os.setName(operatingSystem.getOperatingSystemType()
					.getOperatingSystemName().toString());
			os.setVersion(operatingSystem.getOperatingSystemVersion());
			break;
		}
		return os;
	}

	/**
	 * @param ad
	 * @return
	 */
	private ActivityIdentification ensureActivityIdentification(
			ActivityDescription ad) {
		if (ad.getActivityIdentification() == null) {
			return ad.addNewActivityIdentification();
		}
		return ad.getActivityIdentification();
	}

	/**
	 * @param ad
	 * @return
	 */
	private DataStaging ensureDataStaging(ActivityDescription ad) {
		if (ad.getDataStaging() == null) {
			return ad.addNewDataStaging();
		}
		return ad.getDataStaging();
	}

	/**
	 * @param ad
	 * @return
	 */
	private ExecutableType ensureExecutable(ActivityDescription ad) {
		ensureApplication(ad);
		if (ad.getApplication().getExecutable() == null) {
			return ad.getApplication().addNewExecutable();
		}
		return ad.getApplication().getExecutable();
	}

	/**
	 * @param ad
	 * @return
	 */
	private Application ensureApplication(ActivityDescription ad) {
		if (ad.getApplication() == null) {
			return ad.addNewApplication();
		}
		return ad.getApplication();
	}

	/**
	 * @see eu.unicore.hila.job.spi.ModelToNative#getFormatName()
	 */
	public String getFormatName() {
		return EMI_ADL;
	}

}
