/*********************************************************************************
 * Copyright (c) 2012 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.gridftp;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.Security;
import java.util.Properties;

import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLAFactoryException;

/**
 * @author bjoernh
 * 
 *         16.07.2012 14:30:29
 * 
 */
public class GridFtpProperties extends Properties {

	private static final String GRIDFTP_KEYSTORE = "hila2.gridftp.keystore";
	private static final String GRIDFTP_KEYSTORE_PASS = "hila2.gridftp.keystore.password";
	private static final String GRIDFTP_KEYSTORE_ALIAS = "hila2.gridftp.keystore.alias";
	private KeyStore keystore = null;

	static {
		if (Security.getProvider("BC") == null) {
			Security.insertProviderAt(
					new org.bouncycastle.jce.provider.BouncyCastleProvider(), 1);
		}
	}

	private static GridFtpProperties theInstance = null;

	public static synchronized GridFtpProperties getInstance()
			throws HiLAFactoryException {
		if (theInstance == null) {
			theInstance = new GridFtpProperties();
		}
		return theInstance;
	}

	/**
	 * @throws HiLAFactoryException
	 * 
	 */
	public GridFtpProperties() throws HiLAFactoryException {
		try {
			load(new FileInputStream(new File(new File(
					System.getProperty("user.home"), ".hila2"),
					"gridftp.properties")));
		} catch (FileNotFoundException e) {
			throw new HiLAFactoryException(
					"No GridFTP configuration file found.", e);
		} catch (IOException e) {
			throw new HiLAFactoryException(
					"Unable to read GridFTP configuration file.", e);
		}
	}

	/**
	 * @return the keystore
	 * @throws HiLAException
	 */
	public synchronized KeyStore getKeystore() throws HiLAException {
		if (keystore == null) {
			try {
				keystore = KeyStore.getInstance("JKS");
			} catch (KeyStoreException e1) {
				throw new HiLAException("Unable to create keystore instance.",
						e1);
			}
			// catch (NoSuchProviderException e) {
			// throw new HiLAException(
			// "Something is severely wrong. Check your classpath for bcprov library.",
			// e);
			// }
			String ksLocation = getProperty(GRIDFTP_KEYSTORE);
			String ksPass = getProperty(GRIDFTP_KEYSTORE_PASS);
			if (ksLocation == null) {
				throw new HiLAException("Unable to load keystore, as "
						+ GRIDFTP_KEYSTORE + " has not been set.");
			}
			try {
				keystore.load(new FileInputStream(ksLocation),
						ksPass.toCharArray());
			} catch (Exception e) {
				throw new HiLAException("Unable to load keystore", e);
			}
		}
		return keystore;
	}

	/**
	 * @return the password
	 */
	public String getPassword() {
		return getProperty(GRIDFTP_KEYSTORE_PASS);
	}

	/**
	 * @return the alias
	 */
	public String getAlias() {
		return getProperty(GRIDFTP_KEYSTORE_ALIAS);
	}

}
