/*********************************************************************************
 * Copyright (c) 2012 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.gridftp;

import java.io.IOException;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;

import org.apache.log4j.Logger;
import org.globus.ftp.FileInfo;
import org.globus.ftp.GridFTPClient;
import org.globus.ftp.exception.ClientException;
import org.globus.ftp.exception.ServerException;
import org.globus.gsi.GlobusCredential;
import org.globus.gsi.gssapi.GlobusGSSCredentialImpl;
import org.ietf.jgss.GSSCredential;
import org.ietf.jgss.GSSException;

import eu.unicore.hila.Location;
import eu.unicore.hila.LocationPattern;
import eu.unicore.hila.Resource;
import eu.unicore.hila.ResourceTypeRegistry;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseStorage;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLAFactoryException;
import eu.unicore.hila.exceptions.HiLALocationSyntaxException;
import eu.unicore.hila.grid.Storage;

/**
 * BEWARE: The user interface of FTPClient and GridFTPClient is not thread safe
 * [1].
 * 
 * [1] http://www.globus.org/cog/jftp/guide.html
 * 
 * @author bjoernh
 * 
 *         06.02.2012 15:28:45
 * 
 */
@ResourceType(locationStructure = {
		GridFTPImplementationProvider.SCHEME + "://{hostname}:{port}/?",
		GridFTPImplementationProvider.SCHEME + "://{hostname}/?" })
public class GridFTPStorage extends BaseStorage implements Storage {

	private static final Logger log = Logger.getLogger(GridFTPStorage.class);

	private static final Cache storagesCache;
	static {
		final CacheManager manager = CacheManager.getInstance();
		manager.addCache(new Cache(GridFTPStorage.class.getName(), 500, true,
				true, 0, 0));
		storagesCache = manager.getCache(GridFTPStorage.class.getName());
	}

	final Lock clientLock = new ReentrantLock(true);
	private GridFTPClient client;
	private Map<String, String> locationVars;

	/**
	 * @param _location
	 * @throws HiLALocationSyntaxException
	 * @throws HiLAFactoryException
	 */
	private GridFTPStorage(Location _location)
			throws HiLALocationSyntaxException, HiLAFactoryException {
		super(_location);
		try {
			clientLock.lock();

			LocationPattern lp = ResourceTypeRegistry.getInstance()
					.determineLocationPattern(_location);
			locationVars = lp.fillTemplates(_location);
			createClient();
			// client.setDataChannelAuthentication(DataChannelAuthentication.SELF);
			// client.setDataChannelProtection(GridFTPSession.PROTECTION_SAFE);
			// client.setLocalPassive();
			// client.setActive();
			// client.setPassive();
			// client.setLocalActive();
		} catch (ServerException e) {
			throw new HiLAFactoryException(
					"Cannot connect to GridFTP server for " + location, e);
		} catch (IOException e) {
			throw new HiLAFactoryException(
					"Cannot connect to GridFTP server for " + location, e);
		} catch (GSSException e) {
			throw new HiLAFactoryException(
					"Unable to authenticate against GridFTP server.", e);
		} catch (HiLAException e) {
			throw new HiLAFactoryException("Unable to setup GridFTP storage.",
					e);
		} finally {
			clientLock.unlock();
		}
	}

	/**
	 * @throws IOException
	 * @throws ServerException
	 * @throws GSSException
	 * @throws HiLAException
	 */
	private void createClient() throws IOException, ServerException,
			GSSException, HiLAException {
		client = new GridFTPClient(locationVars.get("{hostname}"),
				locationVars.get("{port}") == null ? 2811
						: Integer.parseInt(locationVars.get("{port}")));
		// client.authenticate(new
		// LocalCredentialHelper().getDefaultCredential());

		GridFtpProperties props = GridFtpProperties.getInstance();
		GlobusCredential cred = null;
		try {
			cred = new GlobusCredential(
					(PrivateKey) props.getKeystore().getKey(props.getAlias(),
							props.getPassword().toCharArray()),
					new X509Certificate[] { (X509Certificate) props
							.getKeystore().getCertificate(props.getAlias()) });

			// } catch (HiLAException e) {
			// throw e;

		} catch (Exception e) {
			// throw new HiLAException("Keystore handling exception.", e);
			log.warn("Exception using keystore configuration.", e);
		}

		if (cred != null) {
			GSSCredential gssCred = new GlobusGSSCredentialImpl(cred, 0);
			client.authenticate(gssCred);
		} else {
			client.authenticate(new LocalCredentialHelper()
					.getDefaultCredential());
		}

		client.setProtectionBufferSize(16384);
		client.changeDir("/");
	}

	public static synchronized GridFTPStorage locate(Location _location,
			Object... _extraInformation) throws HiLALocationSyntaxException,
			HiLAFactoryException {
		if (storagesCache.isKeyInCache(_location)) {
			Element storageElem = storagesCache.get(_location);
			return (GridFTPStorage) storageElem.getObjectValue();
		} else {
			GridFTPStorage gftp = new GridFTPStorage(_location);
			storagesCache.put(new Element(_location, gftp));
			return gftp;
		}
	}

	/**
	 * IMPORTANT: caller must ensure that corresponding unlockClient is called,
	 * e.g. with a finally { storage.unlockClient() } block
	 * 
	 * @throws HiLAException
	 */
	public GridFTPClient lockClient() throws HiLAException {
		clientLock.lock();
		try {
			client.getCurrentDir();
		} catch (ServerException e) {
			try {
				createClient();
			} catch (Exception e1) {
				clientLock.unlock();
				throw new HiLAException("Unable to (re)create GridFTP client.",
						e1);
			}
		} catch (IOException e) {
			clientLock.unlock();
			throw new HiLAException("IO Exception in GridFTP client.", e);
		}
		return client;
	}

	public void unlockClient() {
		clientLock.unlock();
	}

	/**
	 * FTP command LIST, unlike most other commands, transfers the data over the
	 * data channel. In that fashion it is very similar to file transfer
	 * commands. FTPClient.list() requires the same sequence of actions as put()
	 * or get(), including setting transfer and server modes if necessary [1].
	 * 
	 * Transfer type must be ASCII.
	 * 
	 * To obtain programmatic access to information of a single remote file, use
	 * FTPClient.mlst() command. The resulting MlsxEntry object will contain all
	 * the information returned by the server.
	 * 
	 * To obtain remote directory listing with similar properties, use
	 * FTPClient.mlsd() command which will return a Vector of MlsxEntry objects,
	 * each corresponding to one file in the directory.
	 * 
	 * @see eu.unicore.hila.common.grid.BaseStorage#getChildren()
	 */
	@Override
	public List<Resource> getChildren() throws HiLAException {
		List<Resource> files = new ArrayList<Resource>();
		try {
			lockClient();
			// client.setMode(GridFTPSession.MODE_STREAM);
			// client.setType(GridFTPSession.TYPE_ASCII);
			client.setPassive();
			client.setLocalActive();
			Vector fileList = client.list();
			for (Object object : fileList) {
				FileInfo file = (FileInfo) object;
				if (file.getName().equals("..") || file.getName().equals(".")) {
					continue;
				}
				Location fileLocation = location.getChildLocation(file
						.getName());
				files.add(fileLocation.locate(this, file));
			}
		} catch (ServerException e) {
			throw new HiLAException("Unable to list file(s) for " + location, e);
		} catch (ClientException e) {
			throw new HiLAException("Unable to list file(s) for " + location, e);
		} catch (IOException e) {
			throw new HiLAException("Unable to list file(s) for " + location, e);
		} finally {
			unlockClient();
		}
		return files;
	}

}
