/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.common.grid;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;

import eu.unicore.hila.HiLA;
import eu.unicore.hila.Location;
import eu.unicore.hila.common.BaseResource;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLANotImplementedException;
import eu.unicore.hila.grid.History;
import eu.unicore.hila.grid.Progressable;
import eu.unicore.hila.grid.StatusChangeListener;
import eu.unicore.hila.grid.StatusChangedEvent;
import eu.unicore.hila.grid.Task;
import eu.unicore.hila.grid.TaskStatus;

/**
 * @author bjoernh
 * 
 *         13.08.2009 10:01:30
 * 
 */
public abstract class BaseTask extends BaseResource implements Task,
	Progressable {

    private static final Logger log = Logger.getLogger(BaseTask.class);

    private TaskStatus currentState = null;
    private List<StatusChangeListener> statusListeners = new ArrayList<StatusChangeListener>();

    private ScheduledFuture<?> monitorFuture;

    protected HiLAException error = null;

    /**
     * @param location
     */
    public BaseTask(Location location) {
	super(location);
    }

    public HiLAException getError() {
	final HiLAException error = this.error;
	this.error = null;
	return error;
    }

	/**
	 * 
	 * @param _status
	 * @return <code>true</code> if the current status is final,
	 *         <code>false</code> otherwise.
	 * @deprecated Use TaskStatus.isFinalState instead.Deprecated in 2.4.0,
	 *             should be removed by 2.5.0 at the latest
	 */
    protected boolean isFinalState(TaskStatus _status) {
	if (_status == null) {
	    return false;
	} else {
	    return _status.equals(TaskStatus.ABORTED)
		    || _status.equals(TaskStatus.FAILED)
		    || _status.equals(TaskStatus.SUCCESSFUL);
	}
    }

    /**
     * @see eu.unicore.hila.grid.Task#registerStatusChangeListener(eu.unicore.hila.grid.StatusChangeListener)
     */
    public void registerStatusChangeListener(StatusChangeListener _listener) {
	statusListeners.add(_listener);
	if (statusListeners.size() == 1) {
	    startPeriodicMonitoring();
	}
    }

    /**
     * 
     */
    private void startPeriodicMonitoring() {
	if (log.isDebugEnabled()) {
	    log.debug("Starting periodic monitoring of Task " + location);
	}

	monitorFuture = HiLA.getDaemonExecutor().scheduleWithFixedDelay(
		new StatusMonitor(this), 0, 1, TimeUnit.SECONDS);
    }

    /**
     * @see eu.unicore.hila.grid.Task#removeStatusChangeListener(eu.unicore.hila.grid.StatusChangeListener)
     */
    public void removeStatusChangeListener(StatusChangeListener _listener) {
	statusListeners.remove(_listener);
	if (statusListeners.size() == 0) {
	    stopPeriodicMonitoring();
	}
    }

    /**
     * 
     */
    private void stopPeriodicMonitoring() {
	if (log.isDebugEnabled()) {
	    log.debug("Stopping periodic monitoring of Task " + location);
	}
	monitorFuture.cancel(false);
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.Task#block()
     */
    public TaskStatus block() throws HiLAException {
	while (!isDone()) {
	    try {
		Thread.sleep(1000);
	    } catch (InterruptedException e) {
		//
	    }
	}
	return status();
    }

    /**
     * @see eu.unicore.hila.grid.Task#status()
     */
    @Override
    public TaskStatus status() throws HiLAException {
	return currentState;
    }

    /**
     * @see eu.unicore.hila.grid.Task#getHistory()
     */
    @Override
    public History getHistory() throws HiLAException {
	throw new HiLANotImplementedException("History not implemented.");
    }

    /**
     * @return
     * @throws HiLAException
     */
    public boolean isDone() throws HiLAException {
		return status().isFinalState();
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.Task#block(long)
     */
    public TaskStatus block(long arg0) throws HiLAException {
		final long deadline = System.currentTimeMillis() + arg0;
		while (!isDone()) {
			if(deadline <= System.currentTimeMillis()) {
				break;
			}
			try {
				Thread.sleep(Math.min(1000,
						deadline - System.currentTimeMillis()));
			} catch (InterruptedException e) {
				//
			}
		}
		return status();
    }

    private class StatusMonitor implements Runnable {

	Task task;

	/**
	 * @param baseTask
	 */
	public StatusMonitor(Task _task) {
	    this.task = _task;
	}

	/**
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run() {
	    try {
		TaskStatus oldStatus = currentState;
		status();
		if (oldStatus != currentState) {
		    notifyListeners(task, currentState);
		}
	    } catch (HiLAException e) {
		log.error("Monitoring of job failed", e);
	    }
	}

    }

    // Progressable implementations
    /**
     * @throws HiLANotImplementedException
     * @see eu.unicore.hila.grid.Progressable#getUnit()
     */
    @Override
    public String getUnit() throws HiLANotImplementedException {
	throw new HiLANotImplementedException(
		"Empty Progressage.getUnit() implementation.");
    }

    /**
     * 
     */
    public void notifyListeners(Task _task, TaskStatus _status) {
	for (StatusChangeListener listener : statusListeners) {
	    listener.statusChanged(new StatusChangedEvent(_task, _status));
	}
    }

    /**
     * @throws HiLANotImplementedException
     * @see eu.unicore.hila.grid.Progressable#getCurrent()
     */
    @Override
    public long getCurrent() throws HiLAException {
	throw new HiLANotImplementedException(
		"Empty Progressable.getCurrent() implementation.");
    }

    /**
     * @throws HiLAException
     * @see eu.unicore.hila.grid.Progressable#getMax()
     */
    @Override
    public long getMax() throws HiLAException {
	throw new HiLANotImplementedException(
		"Empty Progressable.getMax() implementation.");
    }

    protected void setCurrentState(TaskStatus _status, HiLAException e) {
	if (e != null) {
	    this.error = e;
	}
	currentState = _status;
	synchronized (this) {
	    this.notifyAll();
	}
    }

    protected void setCurrentState(TaskStatus _status) {
	setCurrentState(_status, null);
    }
}
