/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;
import java.util.List;

import javax.annotation.processing.FilerException;

import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.exceptions.HiLAException;

/**
 * @author bjoernh
 * 
 *         29.05.2009
 */
public interface File extends Resource {
    /**
     * 
     * @return In case of a directory, the list of files contained therein, in
     *         case of a plain file, the list contains the file itself.
     * @throws HiLAException
     */
    List<File> ls() throws HiLAException;

    /**
     * Whether this file is the root of a storage, i.e. its path inside the
     * storage is the empty path.
     * 
     * @return
     * @throws HiLAException
     */
    boolean isRoot() throws HiLAException;

    /**
     * Whether this {@link File} is a directory. This is only meaningful if the
     * {@link File} {@link #exists()}.
     * 
     * @return true, if the file is a directory, false otherwise
     * @throws HiLAException
     *             if the file does not exist.
     */
    boolean isDirectory() throws HiLAException;

    /**
     * Checks, whether the {@link File} {@link #exists()}.
     * 
     * @return true, if the {@link File} {@link #exists()}, false otherwise
     * @throws HiLAException
     */
    boolean exists() throws HiLAException;

    /**
     * 
     * @return true, if this file is readable for the user.
     * @throws HiLAException
     */
    boolean isReadable() throws HiLAException;

    /**
     * Sets the readable flag of the {@link File} for the user.
     * 
     * @param _readable
     * @throws HiLAException
     */
    void setIsReadable(boolean _readable) throws HiLAException;

    /**
     * 
     * @return true, if this {@link File} is writable for the user.
     * @throws HiLAException
     */
    boolean isWritable() throws HiLAException;

    /**
     * Sets the writable flag of the {@link File} for the user.
     * 
     * @param _writable
     * @throws HiLAException
     */
    void setIsWritable(boolean _writable) throws HiLAException;

    /**
     * 
     * @return true, if the {@link File} is executable for the user.
     * @throws HiLAException
     */
    boolean isExecutable() throws HiLAException;

    /**
     * Sets the executable flag of the {@link File} for the user.
     * 
     * @param _executable
     * @throws HiLAException
     */
    void setIsExecutable(boolean _executable) throws HiLAException;

    /**
     * Sets readable, writable, executable flags all at once.
     * 
     * @param _readable
     * @param _writable
     * @param _executable
     * @throws HiLAException
     */
    void chmod(boolean _readable, boolean _writable, boolean _executable)
	    throws HiLAException;

    /**
     * Deletes the file.
     * 
     * @return true, if the file has been deleted successfully, false otherwise.
     * @throws HiLAException
     */
    boolean delete() throws HiLAException;

    /**
     * 
     * @param _recursive
     * @return
     * @throws HiLAException
     */
    boolean delete(boolean _recursive) throws HiLAException;

    /**
     * Creates a directory at the {@link Location} of this {@link File}.
     * 
     * @return true, if the directory has been created, false otherwise
     * @throws HiLAException
     *             If anything went wrong when creating the directory.
     */
    boolean mkdir() throws HiLAException;

    /**
     * 
     * @param _createParents
     *            whether to recursively create directories and create the
     *            parents
     * @return true, if the directory has been created, false otherwise
     * @throws HiLAException
     *             If anything went wrong when creating the directory.
     */
    boolean mkdir(boolean _createParents) throws HiLAException;

    /**
     * 
     * @return The {@link Storage} that this {@link File} is located in.
     * @throws HiLAException
     */
    Storage getStorage() throws HiLAException;

    /**
     * If the target {@link File} is a directory, then move this
     * {@link java.io.File} into the directory and give the new target the same
     * name.
     * 
     * @param localFile
     *            The local {@link java.io.File} which to import to this remote
     *            {@link File}
     * @return
     * @throws HiLAException
     */
    SimpleTransfer importFromLocalFile(java.io.File localFile)
	    throws HiLAException;

    /**
     * If the target {@link File} is a directory, then move this
     * {@link java.io.File} into the directory and give the new target the same
     * name.
     * 
     * @param localFile
     *            The local {@link java.io.File} which to import to this remote
     *            {@link File}
     * @param _overwrite
     *            whether to overwrite the remote {@link File}
     * @return
     * @throws HiLAException
     */
    SimpleTransfer importFromLocalFile(java.io.File localFile,
	    boolean _overwrite) throws HiLAException;

    /**
     * 
     * @param localFile
     * @param _overwrite
     * @param _recursive
     * @return
     * @throws HiLaException
     */
    SimpleTransfer importFromLocalFile(java.io.File localFile,
	    boolean _overwrite, boolean _recursive) throws HiLAException;

    /**
     * Export this {@link File}. If the target {@link java.io.File} is a
     * directory, then move this {@link File} into the directory and give the
     * new target the same name.
     * 
     * @param localFile
     *            the local {@link java.io.File} to export to
     * @return
     * @throws HiLAException
     */
    SimpleTransfer exportToLocalFile(java.io.File localFile)
	    throws HiLAException;

    /**
     * If the target {@link java.io.File} is a directory, then move this
     * {@link File} into the directory and give the new target the same name.
     * 
     * @param localFile
     *            the local {@link java.io.File} to export to
     * @param _overwrite
     *            whether to overwrite the local {@link java.io.File}
     * @return
     * @throws HiLAException
     */
    SimpleTransfer exportToLocalFile(java.io.File localFile, boolean _overwrite)
	    throws HiLAException;

    /**
     * 
     * @param localFile
     * @param _overwrite
     * @param _recursive
     * @return
     * @throws HiLAException
     */
    SimpleTransfer exportToLocalFile(java.io.File localFile,
	    boolean _overwrite, boolean _recursive) throws HiLAException;

    /**
     * 
     * @return
     * @throws HiLAException
     */
    SimpleTransfer exportToStream(OutputStream _os) throws HiLAException;

    /**
     * 
     * @param _outputStream
     * @throws HiLAException
     */
    SimpleTransfer importFromStream(InputStream _is) throws HiLAException;

    /**
     * 
     * @param _outputStream
     */
    SimpleTransfer importFromStream(InputStream _is, boolean _overwrite)
	    throws HiLAException;

    /**
     * Transfer this {@link File} to _other {@link File}. If the target is a
     * directory, then move this {@link File} into the directory and give the
     * new target the same name.
     * 
     * @param _other
     *            The target {@link File}
     * @return
     * @throws HiLAException
     */
    ThirdPartyTransfer transfer(File _other) throws HiLAException;

    /**
     * Same as {@link #transfer(File)}. If the target is a directory, then move
     * this {@link File} into the directory and give the new target the same
     * name.
     * 
     * @param _other
     *            The target {@link FilerException}
     * @param _overwrite
     *            whether to overwrite the target {@link File}
     * @return
     * @throws HiLAException
     * @see {@link #transfer(File)}
     */
    ThirdPartyTransfer transfer(File _other, boolean _overwrite)
	    throws HiLAException;

    /**
     * Same as {@link #transfer(File)}. If the target is a directory, then move
     * this {@link File} into the directory and give the new target the same
     * name.
     * 
     * @param _other
     *            The target {@link FilerException}
     * @param _overwrite
     *            whether to overwrite the target {@link File}
     * @param _recursive
     *            whether to transfer directory structures recursively
     * @return
     * @throws HiLAException
     * @see {@link #transfer(File)}
     */
    ThirdPartyTransfer transfer(File _other, boolean _overwrite,
	    boolean _recursive) throws HiLAException;

    /**
     * 
     * @return The size of this {@link File} in Bytes.
     * @throws HiLAException
     */
    long size() throws HiLAException;

    /**
     * 
     * @return the date when this {@link File} was last modified
     * @throws HiLAException
     */
    Date lastModified() throws HiLAException;

    /**
     * Copy this {@link File} to _other {@link File}. Only works within a
     * storage. For remote, third-party transfers use {@link #transfer(File)}.
     * If the target is a directory, then move this {@link File} into the
     * directory and give the new target the same name.
     * 
     * @param _other
     *            The target {@link File}.
     * @throws HiLAException
     */
    void copyTo(File _other) throws HiLAException;

    /**
     * Copy this {@link File} to _other {@link File}. Only works within a
     * storage. For remote, third-party transfers use {@link #transfer(File)}.
     * If the target is a directory, then move this {@link File} into the
     * directory and give the new target the same name.
     * 
     * @param _other
     *            The target {@link File}.
     * @param _overwrite
     *            whether to overwrite the target {@link File} if it exists
     * @throws HiLAException
     */
    void copyTo(File _other, boolean _overwrite) throws HiLAException;

    /**
     * Copy this {@link File} to _other {@link File}. Only works within a
     * storage. For remote, third-party transfers use {@link #transfer(File)}.
     * If the target is a directory, then move this {@link File} into the
     * directory and give the new target the same name.
     * 
     * @param _other
     *            The target {@link File}.
     * @param _overwrite
     *            whether to overwrite the target {@link File} if it exists
     * @param _recursive
     *            whether to copy directory structures recursively
     */
    void copyTo(File _other, boolean _overwrite, boolean _recursive)
	    throws HiLAException;

    /**
     * Move this {@link File} to _other {@link File} within a storage. If the
     * target is a directory, then move this {@link File} into the directory and
     * give the new target the same name.
     * 
     * @param _other
     *            The target {@link File} to move to.
     * @throws HiLAException
     */
    void moveTo(File _other) throws HiLAException;

    /**
     * Move this {@link File} to _other {@link File} within a storage. If the
     * target is a directory, then move this {@link File} into the directory and
     * give the new target the same name.
     * 
     * @param _other
     *            The target {@link File} to move to.
     * @param _overwrite
     *            whether to overwrite an existing file.
     * @throws HiLAException
     */
    void moveTo(File _other, boolean _overwrite) throws HiLAException;

}
