/*********************************************************************************
 * Copyright (c) 2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.emi.es;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map.Entry;

import org.apache.log4j.Logger;

import eu.unicore.hila.Location;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseGrid;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLAFactoryException;
import eu.unicore.hila.grid.Config;

/**
 * @author bjoernh
 * 
 *         21.12.2011 11:24:22
 * 
 */
@ResourceType(locationStructure = { EmiEsImplementationProvider.SCHEME + ":/" })
public class EmiEsGrid extends BaseGrid {

   private static final Logger log = Logger.getLogger(EmiEsGrid.class);

   private static final String CONFIG_KEY = "hila.emi.es.config";

   private final EmiEsProperties properties = new EmiEsProperties();

   private Object configLock = new Object();
   private Config config = null;

   /**
    * @param _location
    * @param _extraInformation
    * @throws HiLAException
    */
   private EmiEsGrid(Location _location, Object[] _extraInformation)
         throws HiLAException {
      super(_location);
      initialize();
   }

   public static final EmiEsGrid locate(Location _location,
         Object... _extraInformation) throws HiLAException {
      return new EmiEsGrid(_location, _extraInformation);
   }

   /**
    * 
    */
   public EmiEsProperties getProperties() {
      return properties;
   }

   /**
    * Search order of configuration file:
    * <ol>
    * <li>File referenced by system property {@link EmiEsGrid#CONFIG_KEY}</li>
    * <li>File emi-es.properties in current directory (if exists)</li>
    * <li>File ~/.hila2/emi-es.properties in user home</li>
    * <li>Classpath resource /emi-es.properties</li>
    * </ol>
    * 
    * TODO externalize to hila-grid-common or even hila-api, config loading
    * should be similar for most implementations
    * 
    * @throws HiLAException
    * 
    */
   private void initialize() throws HiLAException {

      InputStream configIS = null;
      try {
         String config = System.getProperty(CONFIG_KEY);
         if (config != null) {
            if (log.isDebugEnabled()) {
               log.debug("Using EMI ES configuration as defined by system property "
                     + CONFIG_KEY + ": " + config);
            }
            File configFile = new File(config);
            configIS = new FileInputStream(configFile);
            properties.load(configIS);
            properties.setConfigBase(configFile.getParentFile()
                  .getAbsoluteFile());
            return;
         }
      } catch (FileNotFoundException e) {
         throw new HiLAFactoryException("Cannot find configuration file.", e);
      } catch (IOException e) {
         throw new HiLAFactoryException("Cannot open configuration file.", e);
      } finally {
         if (configIS != null) {
            try {
               configIS.close();
            } catch (IOException e) {
               log.warn("Unable to close config file input stream.", e);
            }
         }
      }

      try {
         File configFile = new File("emi-es.properties");
         configIS = new FileInputStream(configFile);
         properties.load(configIS);
         properties.setConfigBase(configFile.getParentFile().getAbsoluteFile());
         return;
      } catch (FileNotFoundException e) {
         // this can safely be ignored
      } catch (IOException e) {
         throw new HiLAException(
               "Cannot read configuration file emi-es.properties from current directory.",
               e);
      } finally {
         if (configIS != null) {
            try {
               configIS.close();
            } catch (IOException e) {
               log.warn("Unable to close config file input stream", e);
            }
         }
      }

      try {
         File configFile = new File(System.getProperty("user.home")
               + File.separator + ".hila2" + File.separator
               + "emi-es.properties");
         configIS = new FileInputStream(configFile);
         properties.load(configIS);
         properties.setConfigBase(configFile.getParentFile().getAbsoluteFile());
         return;
      } catch (FileNotFoundException e) {
         // this can safely be ignored
      } catch (IOException e) {
         throw new HiLAException(
               "Cannot read configuration file emi-es.properties from ~/.hila2.",
               e);
      } finally {
         if (configIS != null) {
            try {
               configIS.close();
            } catch (IOException e) {
               log.warn("Unable to close config file input stream", e);
            }
         }
      }

      try {
         configIS = getClass().getResourceAsStream("/emi-es.properties");
         if (configIS == null) {
            throw new HiLAFactoryException(
                  "Unable to find useable EMI ES configuration.");
         } else {
            properties.load(configIS);
         }
      } catch (IOException e) {
         throw new HiLAFactoryException(
               "Unable to read emi-es.properties from classpath", e);
      } finally {
         if (configIS != null) {
            try {
               configIS.close();
            } catch (IOException e) {
               log.warn("Unable to close config file input stream", e);
            }
         }
      }

      if (log.isTraceEnabled()) {
         for (Entry<Object, Object> entry : properties.entrySet()) {
            log.trace(entry.getKey() + " : " + entry.getValue());
         }
      }

   }

   /**
    * @return the config
    */
   public Config getConfig() {
      synchronized (configLock) {
         if (config == null) {
            config = properties.getConfig();
         }
      }
      return config;
   }

}
