/*********************************************************************************
 * Copyright (c) 2012 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.prefs.Preferences;

import org.apache.log4j.Logger;

/**
 * @author bjoernh
 * 
 *         07.02.2012 16:19:17
 * 
 */
public class HiLA {

   private static final Logger log = Logger.getLogger(HiLA.class);

   private static final String COREPOOLSIZE = "corepoolsize";

   private static final String MAXTHREADSSIZE = "maxthreads";

   private static final String THREADTIMEOUT = "threadtimeout";

   protected static final String WAITFORPENDINGTASKS = "waitforpendingtasks";

   private final ScheduledExecutorService daemonExecutor;
   private final ThreadPoolExecutor executor;
   private static HiLA theInstance = null;
   private final Preferences preferences;

   /**
    * 
    */
   private HiLA() {
      preferences = Preferences.userNodeForPackage(HiLA.class);

      daemonExecutor = Executors.newScheduledThreadPool(
            preferences.getInt(COREPOOLSIZE, 5), new ThreadFactory() {

               int threadNum = 0;

               @Override
               public Thread newThread(Runnable r) {
                  Thread t = new Thread(r, "HiLA Daemon Thread "
                        + Integer.toString(threadNum++));
                  t.setDaemon(true);
                  return t;
               }
            });

      // must never use less than 5: it's actually two because of some race
      // condition in the unicore6 implementation, but we better be safe than
      // sorry
      executor = new ThreadPoolExecutor(Math.max(5,
            preferences.getInt(COREPOOLSIZE, 5)), preferences.getInt(
            MAXTHREADSSIZE, 30), preferences.getInt(THREADTIMEOUT, 10),
            TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(),
            new ThreadFactory() {

               int threadNum = 0;

               @Override
               public Thread newThread(Runnable r) {
                  return new Thread(r, "HiLA Thread "
                        + Integer.toString(threadNum++));
               }
            });
      executor.allowCoreThreadTimeOut(true);

      Runtime.getRuntime().addShutdownHook(new Thread(new Runnable() {

         @Override
         public void run() {
            daemonExecutor.shutdown();
            if (executor.isTerminated()) {
               try {
                  log.info("Waiting for pending tasks to finish.");
                  executor.shutdown();
                  executor.awaitTermination(
                        preferences.getInt(WAITFORPENDINGTASKS, 10),
                        TimeUnit.SECONDS);
               } catch (Exception e) {
                  log.error(
                        "Pending tasks not finished after "
                              + preferences.getInt(WAITFORPENDINGTASKS, 10)
                              + " seconds", e);
               } finally {
                  executor.shutdownNow();
               }
            }
         }
      }));

   }

   public static synchronized ScheduledExecutorService getDaemonExecutor() {
      if (theInstance == null) {
         theInstance = new HiLA();
      }

      return theInstance.daemonExecutor;
   }

   public static synchronized ExecutorService getExecutor() {
      if (theInstance == null) {
         theInstance = new HiLA();
      }

      return theInstance.executor;
   }

   public static synchronized void shutdown() throws InterruptedException {
      if (theInstance == null) {
         return;
      } else {
         theInstance.shutdownInternal();
      }
   }

   /**
    * @throws InterruptedException
    * 
    */
   private void shutdownInternal() throws InterruptedException {
      executor.shutdown();
      if (!executor.awaitTermination(
            preferences.getInt(WAITFORPENDINGTASKS, 10), TimeUnit.SECONDS)) {
         log.warn("There are still pending tasks.");
      }
   }
}
