/*********************************************************************************
 * Copyright (c) 2010-2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Random;
import java.util.UUID;

import junit.framework.Assert;

import org.apache.log4j.Logger;
import org.apache.tools.ant.filters.StringInputStream;
import org.junit.Test;

import eu.unicore.bugsreporter.annotation.RegressionTest;
import eu.unicore.hila.Location;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLALocationSyntaxException;
import eu.unicore.hila.grid.File;
import eu.unicore.hila.grid.SimpleTransfer;
import eu.unicore.hila.grid.Site;
import eu.unicore.hila.grid.Storage;
import eu.unicore.hila.grid.Task;
import eu.unicore.hila.grid.TaskStatus;
import eu.unicore.hila.grid.ThirdPartyTransfer;

/**
 * @author bjoernh
 * 
 *         16.07.2010 15:47:03
 * 
 */
public class Unicore6FileTest extends AbstractUnicore6Test {

    private static final Logger log = Logger.getLogger(Unicore6FileTest.class);

    @Test
    public void testFileRoundTrip() throws HiLALocationSyntaxException,
	    HiLAException, IOException {
	Site site = (Site) new Location(siteLoc).locate();
	File storageFile = site.getStorages().get(0).asFile();
	Assert.assertTrue(storageFile.isRoot());
	Assert.assertTrue(storageFile.isDirectory());
	String fileUUID = UUID.randomUUID().toString();
	File remoteFile = (File) storageFile.getChild(fileUUID);
	Assert.assertFalse(remoteFile.isRoot());
	Assert.assertEquals(fileUUID, remoteFile.getName());
	try {
	    Assert.assertFalse(remoteFile.isDirectory());
	    Assert.fail("Should have thrown an exception as this "
		    + "cannot be determined.");
	} catch (HiLAException e) {
	    // this is ok
	}

	java.io.File localFile = createRandomFile();

	SimpleTransfer fileImport = remoteFile.importFromLocalFile(localFile);
	fileImport.block();
	if (!fileImport.status().equals(TaskStatus.SUCCESSFUL)) {
	    Assert.fail("File import status: " + fileImport.status().toString());
	}

	Assert.assertEquals(localFile.length(), remoteFile.size());
	Assert.assertTrue(remoteFile.exists());
	Assert.assertEquals(fileUUID, remoteFile.getName());
	File remoteFile2 = (File) storageFile.getChild(UUID.randomUUID()
		.toString());
	remoteFile.copyTo(remoteFile2);
	try {
	    remoteFile.copyTo(remoteFile2);
	    Assert.fail("Should have thrown an exception, because"
		    + " files should only be overwritten explicitly.");
	} catch (HiLAException e) {
	    // this is fine
	}

	remoteFile.copyTo(remoteFile2, true);
	Assert.assertEquals(remoteFile.size(), remoteFile2.size());

	remoteFile.delete();
	Assert.assertFalse(remoteFile.exists());
	remoteFile2.delete();
	Assert.assertFalse(remoteFile2.exists());

	try {
	    remoteFile.size();
	    Assert.fail("should have thrown an exception because of non existing file");
	} catch (HiLAException e) {

	}

	localFile.delete();
    }

    /**
     * @return
     * @throws IOException
     * @throws FileNotFoundException
     */
    private java.io.File createRandomFile() throws IOException,
	    FileNotFoundException {
	java.io.File localFile = java.io.File
		.createTempFile("hila_test_", null);

	OutputStream os = new FileOutputStream(localFile);
	Random rnd = new Random();
	for (int i = 0; i < 100; ++i) {
	    os.write(rnd.nextInt());
	}
	os.close();
	return localFile;
    }

    @Test
    public void testImportIntoDirectory() throws HiLALocationSyntaxException,
	    HiLAException, FileNotFoundException, IOException {
	Site site = (Site) new Location(siteLoc).locate();
	File storageFile = site.getStorages().get(0).asFile();

	Assert.assertTrue(storageFile.exists());

	java.io.File localFile = createRandomFile();

	storageFile.importFromLocalFile(localFile).block();

	File fileInStorage = (File) storageFile.getChild(localFile.getName());
	Assert.assertTrue(fileInStorage.exists());

	Assert.assertTrue(fileInStorage.delete());

	Assert.assertTrue(localFile.delete());
    }

    @Test
    public void testImportFileFromStream() throws FileNotFoundException,
	    IOException, HiLALocationSyntaxException, HiLAException {
	java.io.File localFile = createRandomFile();

	InputStream is = new FileInputStream(localFile);

	Site site = (Site) new Location(siteLoc).locate();
	File storageFile = site.getStorages().get(0).asFile();
	File fileInStorage = (File) storageFile.getChild(UUID.randomUUID()
		.toString());

	SimpleTransfer st = fileInStorage.importFromStream(is);
	TaskStatus status = st.block();
	Assert.assertEquals(TaskStatus.SUCCESSFUL, status);

	java.io.File localExportFile = java.io.File
		.createTempFile("hila", null);
	st = fileInStorage.exportToLocalFile(localExportFile, true);
	status = st.block();
	Assert.assertEquals(TaskStatus.SUCCESSFUL, status);

	Assert.assertTrue(fileInStorage.delete());
    }

    @Test
    public void testImportExistingFileFromStream()
	    throws HiLALocationSyntaxException, HiLAException {
	String string = "This is a test.";
	Site site = (Site) new Location(siteLoc).locate();
	File remoteFile = (File) site.getStorages().get(0).asFile()
		.getChild(UUID.randomUUID().toString());
	remoteFile.importFromStream(new StringInputStream(string)).block();

	try {
	    remoteFile.importFromStream(new StringInputStream(string)).block();
	    Assert.fail("Should have thrown an exception,"
		    + " as the file already exists.");
	} catch (HiLAException e) {
	    log.info("Exception thrown as expected.", e);
	}

	remoteFile.importFromStream(new StringInputStream(string), true)
		.block();

	Assert.assertTrue(remoteFile.delete());
    }

    // @Test(expected = HiLAException.class)
    public void testFileInUnlocatableStorage()
	    throws HiLALocationSyntaxException, HiLAException {
	new Location(siteLoc).getChildLocation("storages")
		.getChildLocation("unknown").getChildLocation("files")
		.getChildLocation("test").locate();
    }

    @Test
    public void testGetCachedFile() throws HiLALocationSyntaxException,
	    HiLAException {
	Site site = (Site) new Location(siteLoc).locate();
	File file1 = (File) site.getStorages().get(0).asFile()
		.getChild("blurp");
	File file2 = (File) file1.getLocation().locate();

	Assert.assertSame(file1, file2);
    }

    @Test
    public void testDeleteFile() throws HiLALocationSyntaxException,
	    HiLAException {
	Site site = (Site) new Location(siteLoc).locate();
	File file = (File) site.getStorages().get(0).asFile()
		.getChild(UUID.randomUUID().toString());
	String testString = "This is a test.";
	InputStream is = new StringInputStream(testString);
	file.importFromStream(is).block();
	try {
	    is.close();
	} catch (IOException e) {
	    // ignore
	}
	Assert.assertTrue(file.exists());
	Assert.assertEquals(testString.length(), file.size());
	Assert.assertTrue(file.delete());

	try {
	    Thread.sleep(1000);
	} catch (InterruptedException e) {
	}

	file.mkdir();
	Assert.assertTrue(file.isDirectory());
	try {
	    file.delete();
	    Assert.fail("Should have thrown an exception. delete() "
		    + "should only delete directories in recursive mode");
	} catch (HiLAException e) {
	    log.info("Expected exception thrown.");
	}

	Assert.assertTrue(file.delete(true));

	try {
	    file.delete();
	    Assert.fail("Should have thrown an exception. We just deleted"
		    + " that file, so it cannot be deleted again.");
	} catch (HiLAException e) {
	    log.info("Expected exception thrown.");
	}
    }

    @Test
    public void testExportToExistingLocalDirectoryAndDeleteRecursively()
	    throws HiLALocationSyntaxException, HiLAException {
	Site site = (Site) new Location(siteLoc).locate();
	File file = (File) site.getStorages().get(0).asFile()
		.getChild(UUID.randomUUID().toString());
	file.mkdir();
	File file2 = (File) file.getChild(UUID.randomUUID().toString());
	String test = "This is a test.";
	file2.importFromStream(new StringInputStream(test)).block();
	Assert.assertEquals(test.length(), file2.size());

	try {
	    java.io.File localDir = java.io.File.createTempFile("hila", null);
	    localDir.delete();
	    localDir.mkdir();

	    file2.exportToLocalFile(localDir).block();

	    java.io.File localFile = new java.io.File(localDir, file2.getName());
	    Assert.assertTrue(localFile.delete());
	    Assert.assertTrue(localDir.delete());
	} catch (IOException e) {
	    // TODO Auto-generated catch block
	    log.error("CHANGE ME: bogus error message", e);
	}

	Assert.assertTrue(file.delete(true));
    }

    @Test
    public void testMoveFile() throws HiLALocationSyntaxException,
	    HiLAException {
	String string = "This is a test.";
	Site site = (Site) new Location(siteLoc).locate();
	File file = (File) site.getStorages().get(0).asFile()
		.getChild(UUID.randomUUID().toString());
	Assert.assertEquals(TaskStatus.SUCCESSFUL,
		file.importFromStream(new StringInputStream(string)).block());

	// move simple file (rename)
	String newName = UUID.randomUUID().toString();
	File newFile = (File) file.getParent().getChild(newName);

	Assert.assertFalse(newFile.exists());
	Assert.assertTrue(file.exists());
	file.moveTo(newFile);
	Assert.assertFalse(file.exists());
	Assert.assertTrue(newFile.exists());

	// move file into directory (auto redirect)
	File directory = (File) file.getParent().getChild(
		UUID.randomUUID().toString());
	directory.mkdir();
	Assert.assertTrue(directory.exists());

	newFile.moveTo(directory);
	Assert.assertFalse(newFile.exists());
	Assert.assertTrue(((File) directory.getChild(newFile.getName()))
		.exists());

	// move directory structure (rename)
	File newDirectory = (File) directory.getParent().getChild(
		UUID.randomUUID().toString());
	Assert.assertFalse(newDirectory.exists());
	directory.moveTo(newDirectory);
	Assert.assertFalse(directory.exists());
	Assert.assertTrue(newDirectory.exists());

	// move directory structure into another one (auto redirect)
	File directory3 = (File) directory.getParent().getChild(
		UUID.randomUUID().toString());
	Assert.assertFalse(directory3.exists());
	Assert.assertTrue(directory3.mkdir());
	newDirectory.moveTo(directory3);
	Assert.assertTrue(((File) directory3.getChild(newDirectory.getName()))
		.exists());
	Assert.assertTrue(directory3.delete(true));
    }

    @Test
    public void testCopyFileOutsiteStorage()
	    throws HiLALocationSyntaxException, HiLAException {
	// create simple file for testing
	String string = "This is a test.";
	Site site = (Site) new Location(siteLoc).locate();
	File file = (File) site.getStorages().get(0).asFile()
		.getChild(UUID.randomUUID().toString());
	Assert.assertEquals(TaskStatus.SUCCESSFUL,
		file.importFromStream(new StringInputStream(string)).block());

	Assert.assertEquals("Need at least two storages.", 2, site
		.getStorages().size());

	File s2 = (File) site.getStorages().get(1).asFile()
		.getChild(UUID.randomUUID().toString());
	try {
	    file.copyTo(s2);
	    Assert.fail("Should have thrown an exception");
	} catch (HiLAException e) {
	    log.info("Exception thrown as expected.");
	}

	Assert.assertTrue(file.delete());
    }

    @Test
    public void testMkdirRecursive() throws HiLALocationSyntaxException,
	    HiLAException {
	final File file = ((Site) new Location(siteLoc).locate()).getStorages()
		.get(0).asFile();
	String topLevel = UUID.randomUUID().toString();
	final File file2 = (File) file.getChild(topLevel).getChild("test1")
		.getChild("test2");
	file2.mkdir(true);
	Assert.assertTrue(file2.exists());
	final File file3 = (File) file.getChild(topLevel);
	Assert.assertTrue(file3.delete(true));
    }

    @Test
    public void testCopyDirectoryStructure()
	    throws HiLALocationSyntaxException, HiLAException {
	File directory = (File) ((Site) new Location(siteLoc).locate())
		.getStorages().get(0).asFile()
		.getChild(UUID.randomUUID().toString());
	directory.mkdir();

	File file = (File) directory.getChild(UUID.randomUUID().toString());
	Assert.assertEquals(TaskStatus.SUCCESSFUL,
		file.importFromStream(new StringInputStream("")).block());
	Assert.assertEquals(0, file.size());

	File directory2 = (File) directory.getParent().getChild(
		UUID.randomUUID().toString());

	directory.copyTo(directory2, false, true);

	Assert.assertTrue(directory2.exists());
	Assert.assertTrue(((File) directory2.getChild(file.getName())).exists());

	directory2.delete(true);

	try {
	    directory.copyTo(directory2, false, false);
	    Assert.fail("Should have thrown an exception.");
	} catch (HiLAException e) {
	    log.info("Exception thrown as expected");
	}

	directory2.mkdir();

	directory.copyTo(directory2, false, true);

	Assert.assertTrue(((File) directory2.getChild(directory.getName()))
		.exists());
	Assert.assertTrue(((File) directory2.getChild(directory.getName())
		.getChild(file.getName())).exists());

	directory.delete(true);
	directory2.delete(true);
    }

    @Test
    public void testLastModified() throws HiLALocationSyntaxException,
	    HiLAException {
	// create simple file for testing
	String string = "This is a test.";
	Site site = (Site) new Location(siteLoc).locate();
	File file = (File) site.getStorages().get(0).asFile()
		.getChild(UUID.randomUUID().toString());
	Assert.assertEquals(TaskStatus.SUCCESSFUL,
		file.importFromStream(new StringInputStream(string)).block());

	Calendar lastMod = Calendar.getInstance();
	lastMod.setTime(file.lastModified());
	// limit the allowed deviation to 2 seconds
	Assert.assertEquals((double) Calendar.getInstance().getTimeInMillis(),
		lastMod.getTimeInMillis(), 2000);

	Assert.assertTrue(file.delete());
    }

    @Test
    public void testChmod() throws HiLALocationSyntaxException, HiLAException {
	// create simple file for testing
	String string = "This is a test.";
	Site site = (Site) new Location(siteLoc).locate();
	File file = (File) site.getStorages().get(0).asFile()
		.getChild(UUID.randomUUID().toString());
	Assert.assertEquals(TaskStatus.SUCCESSFUL,
		file.importFromStream(new StringInputStream(string)).block());

	file.chmod(false, false, false);

	Assert.assertFalse(file.isReadable());
	Assert.assertFalse(file.isWritable());
	Assert.assertFalse(file.isExecutable());

	file.setIsReadable(true);
	file.setIsWritable(true);
	file.setIsExecutable(true);

	Assert.assertTrue(file.isReadable());
	Assert.assertTrue(file.isWritable());
	Assert.assertTrue(file.isExecutable());

	file.delete();
    }

    @Test
    public void testTransferSingleFile() throws HiLALocationSyntaxException,
	    HiLAException {
	// create simple file for testing
	String string = "This is a test.";
	Site site = (Site) new Location(siteLoc).locate();
	File file = (File) site.getStorages().get(0).asFile()
		.getChild(UUID.randomUUID().toString());
	Assert.assertEquals(TaskStatus.SUCCESSFUL,
		file.importFromStream(new StringInputStream(string)).block());

	File storage2File = site.getStorages().get(1).asFile();
	ThirdPartyTransfer transfer = file.transfer(storage2File);
	Assert.assertEquals(TaskStatus.SUCCESSFUL, transfer.block());

	transfer.getLocation().locate();

	Assert.assertTrue(storage2File.exists());

	try {
	    Assert.assertEquals(TaskStatus.FAILED, file.transfer(storage2File)
		    .block());
	    Assert.fail("Should have thrown an exception.");
	} catch (HiLAException e) {
	    log.info("Exception thrown as expected.");
	}

	file.delete();
	((File) storage2File.getChild(file.getName())).delete();
    }

    @Test(expected = HiLAException.class)
    public void testDeleteNonExistentFileShouldThrowException()
	    throws HiLALocationSyntaxException, HiLAException {
	Site site = (Site) new Location(siteLoc).locate();
	File file = (File) site.getStorages().get(0).asFile()
		.getChild(UUID.randomUUID().toString());
	file.delete();
    }

    @RegressionTest(date = "2011-02-15", id = 3181843, url = "http://sourceforge.net/tracker/?group_id=102081&func=detail&atid=633902&aid=3181843", description = "Many parallel file exports are unstable in version 2.1")
    @Test
    public void testManyParallelExports() throws HiLALocationSyntaxException,
	    HiLAException, IOException {
	Site site = (Site) new Location(siteLoc).locate();
	Storage storage = site.getStorages().get(0);
	File dir = (File) storage.asFile().getChild(
		UUID.randomUUID().toString());
	dir.mkdir();
	byte[] buf = new byte[178 * 1024];
	Random rnd = new Random();
	rnd.nextBytes(buf);
	List<Task> transfers = new ArrayList<Task>();
	for (int i = 0; i < 100; ++i) {
	    File file = (File) dir.getChild(UUID.randomUUID().toString());
	    ByteArrayInputStream bais = new ByteArrayInputStream(buf);
	    transfers.add(file.importFromStream(bais));
	}
	boolean allDone;
	do {
	    allDone = true;
	    for (Task task : transfers) {
		allDone &= task.status().equals(TaskStatus.ABORTED)
			|| task.status().equals(TaskStatus.FAILED)
			|| task.status().equals(TaskStatus.SUCCESSFUL);
	    }
	} while (!allDone);
	boolean allSuccessful = true;
	for (Task task : transfers) {
	    allSuccessful &= task.status().equals(TaskStatus.SUCCESSFUL);
	}
	Assert.assertTrue(allSuccessful);

	java.io.File tmpDir = java.io.File.createTempFile("hila-test", null);
	tmpDir.delete();
	tmpDir.mkdir();

	transfers.clear();
	List<File> files = dir.ls();
	for (File file : files) {
	    transfers.add(file.exportToLocalFile(tmpDir));
	}

	do {
	    allDone = true;
	    for (Task task : transfers) {
		allDone &= task.status().equals(TaskStatus.ABORTED)
			|| task.status().equals(TaskStatus.FAILED)
			|| task.status().equals(TaskStatus.SUCCESSFUL);
	    }
	} while (!allDone);
	allSuccessful = true;
	for (Task task : transfers) {
	    allSuccessful &= task.status().equals(TaskStatus.SUCCESSFUL);
	}
	Assert.assertTrue(allSuccessful);

	// clean up
	Assert.assertTrue(dir.delete(true));
	java.io.File[] localFiles = tmpDir.listFiles();
	for (java.io.File file : localFiles) {
	    file.delete();
	}
	tmpDir.delete();
    }
}