/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;

import org.apache.log4j.Logger;

import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseGrid;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLAFactoryException;
import eu.unicore.hila.grid.Grid;

/**
 * @author bjoernh
 * 
 *         13.08.2009 09:53:11
 * 
 */
@ResourceType(locationStructure = { "unicore6:/" })
public class Unicore6Grid extends BaseGrid implements Grid {
    private static final Logger log = Logger.getLogger(Unicore6Grid.class);

    private static volatile Unicore6Grid theInstance = null;
    private static final Object theInstanceLock = new Object();

    public static final String CONFIG_KEY = "hila2.unicore6.config";

    private final Unicore6Properties properties = new Unicore6Properties();

   /**
    * @throws HiLAException
    * 
    */
    private Unicore6Grid(Location _location) throws HiLAException {
	super(_location);
	initialize();
    }

    /**
    * Search order of configuration file:
    * <ol>
    * <li>File referenced by system property {@link Unicore6Grid#CONFIG_KEY}</li>
    * <li>File unicore6.properties in current directory (if exists)</li>
    * <li>File ~/.hila2/unicore6.properties in user home</li>
    * <li>Classpath resource /unicore6.properties</li>
    * </ol>
    * 
    * TODO externalize to hila-grid-common or even hila-api, config loading
    * should be similar for most implementations
    * 
    * @throws HiLAException
    * 
    */
    private void initialize() throws HiLAException {

	InputStream configIS = null;
	try {
	    String config = System.getProperty(CONFIG_KEY);
	    if (config != null) {
		if (log.isDebugEnabled()) {
		    log
			    .debug("Using UNICORE 6 configuration as defined by system property "
				    + CONFIG_KEY + ": " + config);
		}
		File configFile = new File(config);
		configIS = new FileInputStream(configFile);
		properties.load(configIS);
		properties.setConfigBase(configFile.getParentFile().getAbsoluteFile());
		return;
	    }
	} catch (FileNotFoundException e) {
	    throw new HiLAFactoryException("Cannot find configuration file.", e);
	} catch (IOException e) {
	    throw new HiLAFactoryException("Cannot open configuration file.", e);
	} finally {
	    if (configIS != null) {
		try {
		    configIS.close();
		} catch (IOException e) {
		    log.warn("Unable to close config file input stream.", e);
		}
	    }
	}

	try {
	    File configFile = new File("unicore6.properties");
	    configIS = new FileInputStream(configFile);
	    properties.load(configIS);
	    properties.setConfigBase(configFile.getParentFile().getAbsoluteFile());
	    return;
	} catch (FileNotFoundException e) {
	    // this can safely be ignored
	} catch (IOException e) {
	    throw new HiLAException(
		    "Cannot read configuration file unicore6.properties from current directory.",
		    e);
	} finally {
	    if (configIS != null) {
		try {
		    configIS.close();
		} catch (IOException e) {
		    log.warn("Unable to close config file input stream", e);
		}
	    }
	}

	try {
	    File configFile = new File(System.getProperty("user.home")
		    + File.separator + ".hila2" + File.separator
		    + "unicore6.properties");
	    configIS = new FileInputStream(configFile);
	    properties.load(configIS);
	    properties.setConfigBase(configFile.getParentFile().getAbsoluteFile());
	    return;
	} catch (FileNotFoundException e) {
	    // this can safely be ignored
	} catch (IOException e) {
	    throw new HiLAException(
		    "Cannot read configuration file unicore6.properties from ~/.hila2.",
		    e);
	} finally {
	    if (configIS != null) {
		try {
		    configIS.close();
		} catch (IOException e) {
		    log.warn("Unable to close config file input stream", e);
		}
	    }
	}

	try {
	    configIS = getClass().getResourceAsStream("/unicore6.properties");
	    if (configIS == null) {
		throw new HiLAFactoryException(
			"Unable to find useable UNICORE 6 configuration.");
	    } else {
		properties.load(configIS);
	    }
	} catch (IOException e) {
	    throw new HiLAFactoryException(
		    "Unable to read unicore6.properties from classpath", e);
	} finally {
	    if (configIS != null) {
		try {
		    configIS.close();
		} catch (IOException e) {
		    log.warn("Unable to close config file input stream", e);
		}
	    }
	}

	if (log.isTraceEnabled()) {
	    for (Entry<Object, Object> entry : properties.entrySet()) {
		log.trace(entry.getKey() + " : " + entry.getValue());
	    }
	}

    }

    /**
     * 
     * @param _location
     * @return
     * @throws HiLAException
     */
    public static Unicore6Grid locate(Location _location) throws HiLAException {
	if (theInstance == null) {
	    synchronized (theInstanceLock) {
		if (theInstance == null) {
		    theInstance = new Unicore6Grid(_location);
		}
	    }
	}
	return theInstance;
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseGrid#getChildren()
     */
    @Override
    public List<Resource> getChildren() throws HiLAException {
	final List<Resource> children = new ArrayList<Resource>();

	children.add(this.location.getChildLocation("sites").locate());
	children.add(this.location.getChildLocation("storages").locate());
	children.add(this.location.getChildLocation("orchestrators").locate());

	return children;
    }

    /**
     * 
     */
    public Unicore6Properties getProperties() {
	return properties;
    }

}
