/*********************************************************************************
 * Copyright (c) 2010 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.shell.completors;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jline.ArgumentCompletor;
import jline.Completor;
import jline.SimpleCompletor;

import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.log4j.Logger;

import eu.unicore.hila.shell.ShellContext;
import eu.unicore.hila.shell.commands.ShellCommand;

/**
 * @author bjoernh
 * 
 *         27.04.2010 12:31:25
 * 
 */
public class CommandMultiplexedCompletor implements Completor {

    private static final Logger log = Logger
	    .getLogger(CommandMultiplexedCompletor.class);

    final private ShellContext shellContext;
    final private Map<String, Completor> commandCompletors;

    /**
     * 
     */
    public CommandMultiplexedCompletor(ShellContext _ctx) {
	this.shellContext = _ctx;
	commandCompletors = new HashMap<String, Completor>();
    }

    /**
     * @see jline.Completor#complete(java.lang.String, int, java.util.List)
     */
    @Override
    public int complete(String _buffer, int _cursor, List _candidates) {
	if (log.isDebugEnabled()) {
	    log.debug("_buffer: " + _buffer);
	    log.debug("_cursor: " + _cursor);
	    log.debug("_candis: " + _candidates);
	}

	String tBuffer = _buffer.trim();
	String[] bufferEles = tBuffer.split(" ");
	if (bufferEles.length > 0) {
	    String currentCommand = bufferEles[0];
	    if (log.isDebugEnabled()) {
		log.debug("current command is: " + currentCommand);
	    }
	    ShellCommand cmd = shellContext.getCommand(currentCommand);
	    if (cmd != null) {
		if (log.isDebugEnabled()) {
		    log.debug("command object from context: " + cmd);
		}
		Completor cmdCompletor = getCommandCompletor(cmd);
		if (cmdCompletor != null) {
		    if (log.isDebugEnabled()) {
			log.debug("cmdCompletor: " + cmdCompletor);
		    }
		    return cmdCompletor.complete(_buffer, _cursor, _candidates);
		}
	    }
	    // don't do anything if not a known command
	}
	return 0;
    }

    /**
     * @param cmd
     * @return
     */
    private Completor getCommandCompletor(ShellCommand cmd) {
	Completor commandCompletor = commandCompletors.get(cmd.getName());
	if (commandCompletor == null) {
	    Options cmdOpts = cmd.getOptions();
	    if (cmdOpts != null) {
		List<String> optionStrings = new ArrayList<String>();
		for (Option option : (Collection<Option>) cmdOpts.getOptions()) {
		    optionStrings.add("-" + option.getOpt());
		    optionStrings.add("--" + option.getLongOpt());
		}
		if (log.isDebugEnabled()) {
		    log.debug(optionStrings);
		}
		commandCompletor = new ArgumentCompletor(
			new Completor[] { new SimpleCompletor(optionStrings
				.toArray(new String[optionStrings.size()])) });
		commandCompletors.put(cmd.getName(), commandCompletor);
	    }
	}
	return commandCompletor;
    }
}
