/*********************************************************************************
 * Copyright (c) 2010 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.shell.commands;

import java.util.List;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.ParseException;
import org.apache.log4j.Logger;

import eu.unicore.hila.Location;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.grid.File;
import eu.unicore.hila.shell.ShellContext;

/**
 * mkdir -- create a directory
 * 
 * operates in two modes (1) create given remote directories (2) create current
 * remote directory if no file or directory exists in that location
 * 
 * parent directories can be created if explicitly stated (-p)
 * 
 * @author bjoernh
 * 
 *         23.03.2010 13:19:52
 * 
 */
public class MkDir extends ShellCommandBase implements ShellCommand {
    private static final Logger log = Logger.getLogger(MkDir.class);

    /**
     * 
     */
    public MkDir() {
	initializeOptions();
    }

    /**
     * 
     */
    private void initializeOptions() {
	options.addOption(OptionBuilder.withArgName("h").withDescription(
		"Print this help message.").withLongOpt("help").create("h"));
	options.addOption(OptionBuilder.withArgName("p").withDescription(
		"Create parent directories.").withLongOpt("parents")
		.create("p"));
    }

    /**
     * @see eu.unicore.hila.shell.commands.ShellCommandBase#execute(java.lang.String[],
     *      eu.unicore.hila.shell.ShellContext)
     */
    @Override
    public void execute(String[] cmdArgs, ShellContext ctx)
	    throws HiLAException {
	try {
	    CommandLine cmdLine = parser.parse(options, cmdArgs);
	    
	    if(cmdLine.hasOption("h")) {
		printHelp(ctx);
		return;
	    }
	    
	    final boolean createParents = cmdLine.hasOption("p");
	    
	    List<String> locations = cmdLine.getArgList();
	    locations.remove(0); // remove the command itself
	    for (String locationString : locations) {
		Location loc = new Location(locationString);
		if(!loc.isLocationOfType(File.class)) {
		    loc = ctx.getCurrentLocation().getChildLocation(locationString);
		}
		if(!loc.isLocationOfType(File.class)) {
		    log.error(locationString + " is not a valid relative or absolute File location.");
		} else {
		    File file = (File) loc.locate();
		    try {
			if(file.mkdir(createParents)) {
			    log.info("Successfully created " + file.getLocation().toString());
			} else {
			    log.info("Unable to create directory " + file.getLocation().toString());
			}
		    } catch (HiLAException e) {
			log.error("Couldn't create directory: " + file.getLocation().toString());
		    }
		}
	    }
	    
	} catch (ParseException e) {
	    log.error("Unable to parse the command line.", e);
	}
    }
}
