/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.shell.commands;

import java.io.IOException;
import java.util.Formatter;
import java.util.List;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.ParseException;
import org.apache.log4j.Logger;

import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLALocationSyntaxException;
import eu.unicore.hila.grid.File;
import eu.unicore.hila.grid.Task;
import eu.unicore.hila.shell.ShellContext;

/**
 * @author bjoernh
 * 
 *         16.10.2009 13:56:29
 * 
 */
public class Ls extends ShellCommandBase implements ShellCommand {
   private static final Logger log = Logger.getLogger(Ls.class);

   /**
     * 
     */
   public Ls() {
      super();
      initializeOptions();
   }

   /**
     * 
     */
   private void initializeOptions() {
      options.addOption(OptionBuilder.withArgName("l").withLongOpt("detail")
            .hasArg(false).create("l"));
      options.addOption(OptionBuilder.withArgName("h").withLongOpt("help")
            .create("h"));
   }

   /**
    * @see eu.unicore.hila.shell.commands.ShellCommand#getName()
    */
   public String getName() {
      return "ls";
   }

   /**
    * @see eu.unicore.hila.shell.commands.ShellCommand#execute(eu.unicore.hila.shell.ShellContext)
    */
   public void execute(String[] cmdArgs, ShellContext ctx) {
      try {
         CommandLine cmdLine = parser.parse(options, cmdArgs);
         if (cmdLine.hasOption("h")) {
            printHelp(ctx);
            return;
         }
         List<String> locationStrings = cmdLine.getArgList();
         // the first one will be the command itself
         locationStrings.remove(0);
         if (locationStrings.size() == 0) {
            if (ctx.getCurrentLocation() != null) {
               locationStrings.add(ctx.getCurrentLocation().toString());
            } else {
               log.warn("No location given and no current location.");
               return;
            }
         }

         for (String locactionString : locationStrings) {

            try {
               Location loc = new Location(locactionString);
               Resource res = null;
               try {
                  res = loc.locate();
               } catch (HiLALocationSyntaxException e) {
                  // try absolute location
                  res = ctx.getCurrentLocation()
                        .getChildLocation(locactionString).locate();
               }

		    List<Resource> children = res.getChildren();
               for (Resource resource : children) {
                  printResourceLN(ctx, resource, cmdLine.hasOption("l"));
               }
               ctx.getCs().flushConsole();
            } catch (HiLAException e) {
               log.error("Caught HiLAException.", e);
            } catch (Exception e) {
               try {
                  ctx.getCs().printString(
                        "Caught " + e.toString() + " : " + e.getMessage());
                  ctx.getCs().printNewline();
                  ctx.getCs().flushConsole();
               } catch (IOException e1) {
                  System.err.println("Writing to ConsoleReader failed.");
               }
            }
         }

      } catch (ParseException e2) {
         log.error("Unable to parse command line.", e2);
      }
   }

   /**
    * @param ctx
    * @param resource
    * @param _detailed
    */
   private void printResourceLN(ShellContext ctx, Resource resource,
         boolean _detailed) {
      if (resource instanceof File && _detailed) {
         printFileResource(ctx, (File) resource);
      } else if (resource instanceof Task && _detailed) {
         printTaskResource(ctx, (Task) resource);
      } else {
         printResource(ctx, resource);
      }
      try {
         ctx.getCs().printNewline();
      } catch (IOException e) {
         log.error("Unable to print to console", e);
      }

   }

   /**
    * @param ctx
    * @param resource
    */
   private void printResource(ShellContext ctx, Resource resource) {
      try {
         ctx.getCs().printString(resource.getLocation().toString());
      } catch (IOException e) {
         log.info(resource.getLocation().toString());
      }

   }

   /**
    * Should print sth. like the following:
    * 
    * location status
    * 
    * @param ctx
    * @param task
    */
   private void printTaskResource(ShellContext ctx, Task task) {
      try {
         StringBuilder sb = new StringBuilder();
         sb.append(task.getLocation().toString());
         sb.append("\t").append(task.status().toString());
         try {
            ctx.getCs().printString(sb.toString());
         } catch (IOException e) {
            log.info(task.status().toString());
         }
      } catch (HiLAException e) {
         log.error(task.getLocation().toString() + "\t" + "status unavailable",
               e);
      }

   }

   /**
    * 
    * @param ctx
    * @param file
    */
   private void printFileResource(ShellContext ctx, File file) {
      StringBuilder sb = new StringBuilder();
      Formatter formatter = new Formatter(sb);
      try {
         formatter.format(" %c%c%c%c %12d %s", file.isDirectory() ? 'd' : '-',
               file.isReadable() ? 'r' : '-', file.isWritable() ? 'w' : '-',
               file.isExecutable() ? 'x' : '-', file.size(), file.getLocation()
                     .toString());

         ctx.getCs().printString(sb.toString());
      } catch (HiLAException e) {
         log.error("Unable to query file properties.", e);
      } catch (IOException e) {
         log.info(sb.toString(), e);
      }

   }

   /**
    * @see eu.unicore.hila.shell.commands.ShellCommand#help()
    */
   public String help() {
      StringBuilder sb = new StringBuilder();
      sb.append("ls -- list a location");
      sb.append(System.getProperty("line.separator"));
      // sb.append(System.getProperty("line.separator"));
      sb.append("lists the current location if none is given,");
      sb.append("otherwise the given locations will be listed.");
      sb.append(System.getProperty("line.separator"));
      // sb.append(System.getProperty("line.separator"));
      return sb.append(super.help()).toString();
   }

}
