/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.shell.commands;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.ParseException;
import org.apache.log4j.Logger;

import eu.unicore.hila.Location;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLALocationSyntaxException;
import eu.unicore.hila.grid.SimpleTransfer;
import eu.unicore.hila.shell.ShellContext;

/**
 * @author bjoernh
 * 
 *         16.10.2009 15:54:43
 * 
 */
public class Import extends ShellCommandBase implements ShellCommand {
    private static final Logger log = Logger.getLogger(Import.class);

    /**
     * 
     */
    public Import() {
	super();
	initializeOptions();
    }

    /**
     * 
     */
    private void initializeOptions() {
	options.addOption(OptionBuilder.withArgName("b")
		.withDescription("Block on task.").withLongOpt("block")
		.hasArg(false).create("b"));
	options.addOption(OptionBuilder.withArgName("h")
		.withDescription("Print help message.").withLongOpt("help")
		.create("h"));
	options.addOption(OptionBuilder.withArgName("f")
		.withDescription("Force overwrite").withLongOpt("force")
		.create("f"));
	options.addOption(OptionConstants.RECURSIVE);
    }

    /**
     * @see eu.unicore.hila.shell.commands.ShellCommand#execute(java.lang.String[],
     *      eu.unicore.hila.shell.ShellContext)
     */
    public void execute(String[] cmdArgs, ShellContext ctx) {
	CommandLine cmdline = null;
	try {
	    cmdline = parser.parse(options, cmdArgs);
	    if (cmdline.hasOption("h")) {
		printHelp(ctx);
		return;
	    }
	} catch (ParseException e) {
	    log.error(e);
	    return;
	}

	List<String> args = cmdline.getArgList();
	args.remove(0);
	String targetPath = args.get(args.size() - 1);

	Location targetLoc = null;
	try {
	    targetLoc = new Location(targetPath);
	    if (!targetLoc.isLocationOfType(eu.unicore.hila.grid.File.class)) {
		targetLoc = ctx.getCurrentLocation().getChildLocation(
			targetPath);
	    }
	    if (!targetLoc.isLocationOfType(eu.unicore.hila.grid.File.class)) {
		log.error("Target " + targetPath + " is not of type File");
		return;
	    }
	} catch (HiLALocationSyntaxException e) {
	    log.error("Last argument is not a valid location.", e);
	    return;
	}

	eu.unicore.hila.grid.File target;
	try {
	    target = (eu.unicore.hila.grid.File) targetLoc.locate();
	    if (args.size() > 2 && !target.isDirectory()) {
		log.error("Target must be a directory for multiple import files.");
		return;
	    }
	} catch (HiLAException e) {
	    log.error("Exception while locating import target "
		    + targetLoc.toString());
	    return;
	}

	List<File> importFiles = new ArrayList<File>();
	for (int i = 0; i < args.size() - 1; ++i) {
	    File f = new File(args.get(i));
	    if (!f.isAbsolute()) {
		f = new File(ctx.getCurrentLocalDir(), args.get(i));
	    }
	    if (f.exists()) {
		importFiles.add(f);
	    } else {
		log.error("Import file " + f.getPath() + " does not exist.");
	    }
	}

	List<SimpleTransfer> importList = new ArrayList<SimpleTransfer>();
	for (File localFile : importFiles) {
	    try {
		importList.add(target.importFromLocalFile(localFile, cmdline
			.hasOption(OptionConstants.FORCE.getArgName()), cmdline
			.hasOption(OptionConstants.RECURSIVE.getArgName())));
	    } catch (HiLAException e) {
		log.error("Unable to import " + localFile + " to "
			+ target.getLocation().toString());
	    }
	}

	for (SimpleTransfer simpleTransfer : importList) {
	    try {
		log.info(simpleTransfer.getLocation().toString() + " "
			+ simpleTransfer.status().toString());
	    } catch (HiLAException e) {
		log.warn("Unable to determine status of transfer "
			+ simpleTransfer.getLocation().toString());
	    }
	}
	
	if(cmdline.hasOption(OptionConstants.BLOCK.getArgName())) {
	    for (SimpleTransfer simpleTransfer : importList) {
		try {
		    simpleTransfer.block();
		} catch (HiLAException e) {
		    log.error("Failed to block on " + simpleTransfer.getLocation());
		}
	    }
	}

    }

    /**
     * @see eu.unicore.hila.shell.commands.ShellCommand#help()
     */
    public String help() {
	StringBuilder sb = new StringBuilder(
		"Import file from the local file system to a remote storage.");
	sb.append(System.getProperty("line.separator"));
	sb.append("import <Local File Path> <Destination File Location>");
	sb.append(System.getProperty("line.separator"));
	return sb.append(super.help()).toString();
    }

}
