/*********************************************************************************
 * Copyright (c) 2008-2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/

package eu.unicore.hila.job.model;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ggf.schemas.jsdl.x2005.x11.jsdl.ResourcesType;

/**
 * A convenience class to help build JSDL documents.
 * 
 * Instantiate an object of this class and use the set and add methods to set or
 * add
 * 
 * <ul>
 * <li>arguments</li>
 * <li>environment variables</li>
 * <li>file exports (after job completion)</li>
 * <li>file imports (before running the job)</li>
 * <li>application name and version</li>
 * <li>executable</li>
 * <li></li>
 * </ul>
 * 
 * Executable and Application (by name and optional version) are mutually
 * exclusive, which is enforced by the methods. Whatever is set last will be
 * used.
 * 
 * @author bjoernh
 * 
 */
public class JobModel {

    private List<Import> imports;

    private List<Export> exports;

    private String taskName;

    // application xor executable
    private String applicationName = null;

    private String applicationVersion = null;
    
    private String applicationDescription = null;

    private String executable = null;

    private List<String> arguments;

    private Map<String, String> environment;

    private long lastModified;

    private ResourcesType resources;

    private List<String> jobAnnotations;

    private String description = null;

    private List<String> jobProjects;

    private List<StageOut> stageouts;

    private List<StageIn> stageIns;

	private String stdout = null;

	private String stderr = null;

	private String stdin = null;

    /**
   * 
   */
    public JobModel() {
	imports = new ArrayList<Import>();
	exports = new ArrayList<Export>();

	arguments = new ArrayList<String>();
	environment = new HashMap<String, String>();
	jobAnnotations = new ArrayList<String>();
	jobProjects = new ArrayList<String>();
	
	stageIns = new ArrayList<StageIn>();
	stageouts = new ArrayList<StageOut>();

	this.lastModified = System.currentTimeMillis();
    }

    /**
     * Get the {@link List} of {@link Import}s already defined.
     * 
     * @return
     */
    public List<Import> getImports() {
	return imports;
    }

    /**
     * Set the {@link List} of {@link Import}s.
     * 
     * @param imports
     */
    public void setImports(List<Import> imports) {
	reset();
	this.imports = imports;
    }

    /**
   * 
   */
    private void reset() {
	this.lastModified = System.currentTimeMillis();
    }
    
    public long getLastModified() {
	return lastModified;
    }

    /**
     * Get the {@link List} of {@link Export}s already defined.
     * 
     * @return
     */
    public List<Export> getExports() {
	return exports;
    }

    /**
     * Set the list of {@link Export}s.
     * 
     * @param exports
     */
    public void setExports(List<Export> exports) {
	reset();
	this.exports = exports;
    }

    /**
     * Get the name of the application currently defined.
     * 
     * @return The name of the application, if it is defined. <code>null</code>
     *         otherwise.
     */
    public String getApplicationName() {
	return applicationName;
    }

    /**
     * Set the name of the application to be used.
     * 
     * @param applicationName
     */
    public void setApplicationName(String applicationName) {
	reset();
	if (applicationName != null) {
	    executable = null;
	    this.applicationName = applicationName;
	}
    }

    /**
     * Get the version of the application to be used.
     * 
     * @return The version of the application, if defined. <code>null</code>
     *         otherwise.
     */
    public String getApplicationVersion() {
	return applicationVersion;
    }

    /**
     * Set the version of the application to be used.
     * 
     * @param applicationVersion
     */
    public void setApplicationVersion(String applicationVersion) {
	reset();
	this.applicationVersion = applicationVersion;
    }
    
    /**
     * 
     * @param _applicationDescription
     */
    public void setApplicationDescription(String _applicationDescription) {
	this.applicationDescription = _applicationDescription;
    }
    
    /**
     * @return the applicationDescription
     */
    public String getApplicationDescription() {
	return applicationDescription;
    }

    public String getExecutable() {
	return executable;
    }

    public void setExecutable(String executable) {
	reset();
	if (executable != null) {
	    applicationName = null;
	    this.executable = executable;
	}
    }

    public List<String> getArguments() {
	return arguments;
    }

    public void setArguments(List<String> arguments) {
	reset();
	this.arguments = arguments;
    }

    public Map<String, String> getEnvironment() {
	return environment;
    }

    public void setEnvironment(Map<String, String> environment) {
	reset();
	this.environment = environment;
    }

    public void setResources(final ResourcesType _resources) {
	reset();
	this.resources = _resources;
    }
    
    public ResourcesType getResources() {
	return this.resources;
    }

    // adders
    public void addArgument(String argument) {
	reset();
	arguments.add(argument);
    }

    public void addEnvironment(String variable, String value) {
	reset();
	environment.put(variable, value);
    }
    
    /**
    * An Export from the working directory of the job to the local file system.
    * 
    * @param _fileName
    * @param _localFile
    */
    public void addExport(String _fileName, File _localFile) {
	reset();
      exports.add(new Export(_fileName, _localFile));
    }

   /**
    * 
    * @param _localFile
    */
   public void addExport(File _localFile) {
      reset();
      exports.add(new Export(_localFile.getName(), _localFile));
   }
    
   /**
    * Import local file into the job's working directory before the job starts.
    * 
    * @param _localFile
    * @param _filename
    */
   public void addImport(File _localFile, String _filename) {
	reset();
      imports.add(new Import(_localFile, _filename));
    }

   /**
    * @see #addImport(File, String)
    * @param _localFile
    */
   public void addImport(File _localFile) {
      addImport(_localFile, _localFile.getName());
    }

 

    public String getTaskName() {
	return taskName;
    }

    public void setTaskName(String _taskName) {
	reset();
	this.taskName = _taskName;
    }

    /**
     * @param _annotation
     */
    public void addJobAnnotation(String _annotation) {
	reset();
	this.jobAnnotations.add(_annotation);
    }

   /**
    * 
    * @return
    */
   public List<String> getJobAnnotations() {
      return this.jobAnnotations;
   }

    /**
     * @param _description
     */
    public void setJobDescription(String _description) {
	reset();
	this.description = _description;
    }
    
    /**
     * @return the description
     */
    public String getJobDescription() {
	return description;
    }

    /**
     * @param project
     */
    public void addJobProject(String _project) {
	this.jobProjects.add(_project);
	
    }

    /**
     * @return
     */
    public List<StageOut> getStageOuts() {
	return stageouts;
    }
    
    public List<StageIn> getStageIns() {
	return stageIns;
    }

    /**
     * @param stageout
     */
    public void addStageOut(StageOut stageout) {
	this.stageouts.add(stageout);
	
    }

    /**
     * @param stageIn
     */
    public void addStageIn(StageIn stageIn) {
	this.stageIns.add(stageIn);
    }

	/**
	 * @return the stdout
	 */
	public String getStdout() {
		return stdout;
	}

	/**
	 * @param stdout
	 *            the stdout to set
	 */
	public void setStdout(String stdout) {
		this.stdout = stdout;
	}

	/**
	 * @return the stderr
	 */
	public String getStderr() {
		return stderr;
	}

	/**
	 * @param stderr
	 *            the stderr to set
	 */
	public void setStderr(String stderr) {
		this.stderr = stderr;
	}

	/**
	 * @return the stdin
	 */
	public String getStdin() {
		return stdin;
	}

	/**
	 * @param stdin
	 *            the stdin to set
	 */
	public void setStdin(String stdin) {
		this.stdin = stdin;
	}

}
