/*********************************************************************************
 * Copyright (c) 2012 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.gridftp;

import java.io.OutputStream;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Callable;

import org.globus.ftp.DataSink;
import org.globus.ftp.DataSinkStream;
import org.globus.ftp.GridFTPClient;

import eu.unicore.hila.HiLA;
import eu.unicore.hila.Resource;
import eu.unicore.hila.common.grid.BaseSimpleTransfer;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLALocationSyntaxException;
import eu.unicore.hila.exceptions.HiLANotImplementedException;
import eu.unicore.hila.grid.SimpleTransfer;
import eu.unicore.hila.grid.TaskStatus;

/**
 * @author bjoernh
 * 
 *         01.03.2012 14:03:35
 * 
 */
public class GridFtpExport extends BaseSimpleTransfer implements
      SimpleTransfer, Callable<TaskStatus> {

   private GridFTPStorage storage;
   private String path;
   private OutputStream os;
   private boolean closeOutputStream;

   /**
    * @param _storage
    * @param _path
    * @param _os
    * @param _closeOutputStream
    * @throws HiLALocationSyntaxException
    */
   public GridFtpExport(GridFTPStorage _storage, String _path,
         OutputStream _os, boolean _closeOutputStream)
         throws HiLALocationSyntaxException {
      super(_storage.getLocation().getChildLocation("exports")
            .getChildLocation(UUID.randomUUID().toString()));
      this.storage = _storage;
      this.path = _path;
      this.os = _os;
      this.closeOutputStream = _closeOutputStream;

      HiLA.getExecutor().submit(this);
   }

   /**
    * @see eu.unicore.hila.grid.Task#abort()
    */
   @Override
   public void abort() throws HiLAException {
      throw new HiLANotImplementedException(
            "abort() not implemented for GridFTP transfers.");
   }

   /**
    * @see eu.unicore.hila.grid.Task#getId()
    */
   @Override
   public String getId() throws HiLAException {
      return getLocation().getName();
   }

   /**
    * @see eu.unicore.hila.Resource#getChildren()
    */
   @Override
   public List<Resource> getChildren() throws HiLAException {
      return Collections.emptyList();
   }

   /**
    * @see java.util.concurrent.Callable#call()
    */
   @Override
   public TaskStatus call() throws Exception {
      setCurrentState(TaskStatus.RUNNING);
      try {
         final GridFTPClient client = storage.lockClient();
         client.setPassive();
         client.setLocalActive();
         // control over os should be up to the caller, so we use
         // NonClosingBufferedOutputStream as an indirection
         DataSink ds = new DataSinkStream(
               new NonClosingBufferedOutputStream(os));
         client.get(path, ds, null);
         // client.close();
         os.flush();
         if (closeOutputStream) {
            os.close();
         }
         setCurrentState(TaskStatus.SUCCESSFUL);
      } catch (Exception e) {
         final HiLAException he = new HiLAException(
               "File export failed.", e);
         setCurrentState(TaskStatus.FAILED, he);
         throw he;
      } finally {
         storage.unlockClient();
      }

      return status();
   }

}
