/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid;

import eu.unicore.hila.Resource;
import eu.unicore.hila.exceptions.HiLAException;

/**
 * A generic {@link Task} interfact, not necessarily representing a
 * {@link Job}.
 * 
 * @author bjoernh
 * 
 *         09.07.2009 10:59:59
 * 
 */
public interface Task extends Resource {

    /**
     * Abort this task.
     * 
     * @throws HiLAException
     */
    void abort() throws HiLAException;

    /**
     * Block on this {@link Task} until it reaches a final state (
     * {@link TaskStatus#SUCCESSFUL}, {@link TaskStatus#ABORTED},
     * {@link TaskStatus#FAILED}).
     * 
     * @return The final {@link TaskStatus}
     * @throws HiLAException
     * @see {@link #block(long)}
     */
    TaskStatus block() throws HiLAException;

    /**
     * Block for at most millis ms until this {@link Task} reaches a final
     * state.
     * 
     * @param millis
     * @return
     * @throws HiLAException
     * @see {@link #block()}
     */
    TaskStatus block(long millis) throws HiLAException;

    /**
     * The reason for the failure of the task can be retrieved here.
     * 
     * @return
     * @throws HiLAException
     */
    HiLAException getError() throws HiLAException;

    /**
     * 
     * @return
     * @throws HiLAException
     */
    History getHistory() throws HiLAException;

    /**
     * The ID of this {@link Task}. Very often, this will be identical to the
     * {@link Task}'s name.
     * 
     * @return
     * @throws HiLAException
     */
    String getId() throws HiLAException;

    /**
     * Register a {@link StatusChangeListener} to be notified of any changes in
     * this {@link Task}'s {@link TaskStatus}.
     * 
     * @param listener
     * @throws HiLAException
     */
    void registerStatusChangeListener(StatusChangeListener listener)
	    throws HiLAException;

    /**
     * Remove a registered {@link StatusChangeListener} from this {@link Task}.
     * 
     * @param listener
     * @throws HiLAException
     */
    void removeStatusChangeListener(StatusChangeListener listener)
	    throws HiLAException;

    /**
     * The current {@link TaskStatus} of this {@link Task}.
     * 
     * @return
     * @throws HiLAException
     */
    TaskStatus status() throws HiLAException;
}
