/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid;

import java.io.File;
import java.util.Calendar;
import java.util.List;

import eu.unicore.hila.exceptions.HiLAException;

/**
 * A {@link Job} is a real computational task that is executed on a compute
 * resources. The distinctive properties are that it can be started with files
 * that can be imported before execution, it has a working directory, stdout,
 * stderr, and an exit status.
 * 
 * @author bjoernh
 * 
 */
public interface Job extends Task {

    /**
     * Synchronous start of the task. This method will return once the
     * {@link Task} is in a final state ({@link TaskStatus#SUCCESSFUL},
     * {@link TaskStatus#ABORTED}, {@link TaskStatus#FAILED}). {@link File}s to
     * be imported into the working directory before execution can be given as
     * arguments.
     * 
     * @param imports
     *            {@link File}s to be imported into the {@link Job}'s working
     *            directory before starting the execution.
     * @return {@link TaskStatus} at the end of execution.
     * @throws HiLAException
     */
    TaskStatus startSync(java.io.File... imports) throws HiLAException;

    /**
     * This method is essentially the same as {@link #startSync(File...)},
     * except that it returns immediately and executes the {@link Task} in the
     * background.
     * 
     * @param imports
     *            {@link File}s to be imported into the {@link Job}'s working
     *            directory before starting the execution.
     * @throws HiLAException
     */
    void startASync(java.io.File... imports) throws HiLAException;

    /**
     * This method is similar to {@link #startASync(File...)}, but an additional
     * parameter allows to provide a {@link StatusChangeListener} that is
     * interested in receiving notifications of changes of the
     * {@link TaskStatus}.
     * 
     * @param listener
     *            A {@link StatusChangeListener} interested in status changes of
     *            the {@link Task}
     * @param imports
     *            {@link File}s to be imported into the {@link Job}'s working
     *            directory before starting the execution.
     * @return {@link TaskStatus} at the time the method returns, this is not
     *         necessarily a final state
     * @throws HiLAException
     */
    TaskStatus startASync(StatusChangeListener listener,
	    java.io.File... imports) throws HiLAException;

    /**
     * Export files from the working directory of the {@link Job} and remove it
     * from its {@link Site}.
     * 
     * @param exports
     *            {@link File}s to be exported before removing the {@link Job}
     *            from the {@link Site}.
     * @return {@link List} of {@link SimpleTransfer} representing the exports.
     * @throws HiLAException
     */
    List<SimpleTransfer> cleanup(eu.unicore.hila.grid.File... exports)
	    throws HiLAException;

    /**
     * The exit code of the {@link Job}.
     * 
     * @return
     * @throws HiLAException
     */
    int getExitCode() throws HiLAException;

    /**
     * A log, if provided by the implementation.
     * 
     * @return
     * @throws HiLAException
     */
    String getLog() throws HiLAException;

    /**
     * The {@link Site} that this {@link Job} is executed at.
     * 
     * @return
     * @throws HiLAException
     */
    Site getSite() throws HiLAException;

    /**
     * A reference to the {@link eu.unicore.hila.grid.File} representing stderr
     * in the working directory of the {@link Job}.
     * 
     * @return
     * @throws HiLAException
     */
    eu.unicore.hila.grid.File getStdErr() throws HiLAException;

    /**
     * A reference to the {@link eu.unicore.hila.grid.File} representing stdout
     * in the working directory of the {@link Job}.
     * 
     * @return
     * @throws HiLAException
     */
    eu.unicore.hila.grid.File getStdOut() throws HiLAException;

    /**
     * 
     * @return A {@link Calendar} object representing the point in time when the
     *         job has been submitted.
     * @throws HiLAException
     */
    Calendar getSubmissionTime() throws HiLAException;

    /**
     * Name of the {@link Job}, potentially more descriptive than its id, but
     * may be mapped to its id by an implementation.
     * 
     * @return
     */
    String getTaskName() throws HiLAException;

    /**
     * The browsable working directory of a {@link Job}. It should be available
     * as soon as the {@link Job} has been created.
     * 
     * @return
     * @throws HiLAException
     */
    eu.unicore.hila.grid.File getWorkingDirectory() throws HiLAException;

    /**
     * Hold the job.
     * 
     * @throws HiLAException
     *             Will be thrown, if the implementation or underlying systems
     *             don't support this operation.
     */
    void hold() throws HiLAException;

    /**
     * Resume a held job.
     * 
     * @throws HiLAException
     *             Will be thrown, if the implementation or underlying systems
     *             don't support this operation.
     */
    void resume() throws HiLAException;

}
