/*********************************************************************************
 * Copyright (c) 2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.bes;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import org.apache.log4j.Logger;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.bes.BESFactory;
import de.fzj.unicore.bes.client.FactoryClient;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import eu.unicore.hila.HiLA;
import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.BaseResource;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLALocationPatternException;
import eu.unicore.hila.grid.Grid;
import eu.unicore.util.httpclient.ClientProperties;

/**
 * @author bjoernh
 * 
 *         16.11.2011 09:31:54
 * 
 */
@ResourceType(locationStructure = { "bes:/sites/?", "bes:/{user}@sites/?" })
public class BesSitesCollection extends BaseResource {
	
	private static final Logger log = Logger.getLogger(BesSitesCollection.class);

	/**
	 * @param _location
	 */
	public BesSitesCollection(Location _location) {
		super(_location);
	}

	/**
	 * @see eu.unicore.hila.Resource#getChildren()
	 */
	@Override
	public List<Resource> getChildren() throws HiLAException {
		List<Resource> sites = new ArrayList<Resource>();

		BesProperties properties = ((BesGrid) getGrid()).getProperties();

		List<String> registryUrls = properties.getRegistryUrls();
		//BesSecurityProperties sp = new BesSecurityProperties(properties);
        ClientProperties sp = new ClientProperties(
                properties);

      ExecutorService executor = HiLA.getExecutor();
		List<Future<List<EndpointReferenceType>>> gsfrFutures = new ArrayList<Future<List<EndpointReferenceType>>>();
		for (String registryUrl : registryUrls) {
			GetServicesFromRegistry gsfr = new GetServicesFromRegistry(
					registryUrl, sp, BESFactory.BES_PORT);
			gsfrFutures.add(executor.submit(gsfr));
		}
		
		for (Future<List<EndpointReferenceType>> future : gsfrFutures) {
			try {
				for (EndpointReferenceType epr : future.get()) {
					try {
						FactoryClient fc = new FactoryClient(epr, sp.clone());
						String siteName = WSServerUtilities.extractFriendlyNameFromEPR(epr);
						if(siteName == null) {
							siteName = new URL(epr.getAddress()
									.getStringValue()).getPath().split("/")[1];
						}
						if (log.isTraceEnabled()) {
							log.trace("siteName is " + siteName);
						}
						Location siteLocation = getLocation().getChildLocation(siteName);
						sites.add(siteLocation.locate(fc));
					} catch (Exception e) {
						log.error("Unable to instantiate FactoryClient for " + epr.getAddress().getStringValue(), e);
					}
				}
			} catch (InterruptedException e) {
				log.error("Interruption while accessing registry.", e);
			} catch (ExecutionException e) {
				log.error("Error while accessing registry.", e);
			}
		}

		return sites;
	}

	/**
	 * @return
	 * @throws HiLAException
	 * @throws HiLALocationPatternException
	 */
	private Resource getGrid() throws HiLALocationPatternException,
			HiLAException {
		return findParentLocationOfType(Grid.class, location, Grid.class)
				.locate();
	}

}
