/*********************************************************************************
 * Copyright (c) 2011 Forschungszentrum Juelich GmbH All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its
 * contributors may be used to endorse or promote products derived from this
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.bes;

import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.ggf.schemas.bes.x2006.x08.besFactory.CreateActivityDocument;
import org.ggf.schemas.bes.x2006.x08.besFactory.CreateActivityResponseDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.bes.client.ActivityClient;
import de.fzj.unicore.bes.client.FactoryClient;
import de.fzj.unicore.bes.faults.InvalidRequestMessageFault;
import de.fzj.unicore.bes.faults.NotAcceptingNewActivitiesFault;
import de.fzj.unicore.bes.faults.UnsupportedFeatureFault;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import eu.unicore.hila.Location;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseSite;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLAFactoryException;
import eu.unicore.hila.exceptions.HiLANotSupportedException;
import eu.unicore.hila.grid.Job;
import eu.unicore.hila.grid.Reservation;
import eu.unicore.hila.grid.Storage;
import eu.unicore.hila.grid.resources.ResourceDescription;
import eu.unicore.hila.job.jsdl.ModelToJsdl;
import eu.unicore.hila.job.model.JobModel;

@ResourceType(locationStructure = {
		BesImplementationProvider.SCHEME + ":/sites/{siteName}/?",
		BesImplementationProvider.SCHEME + ":/{user}@sites/{siteName}/?" })
public class BesSite extends BaseSite {
	private static final Logger log = Logger.getLogger(BesSite.class);

	private FactoryClient factory = null;

	private static final Map<Location, BesSite> sitesCache = new HashMap<Location, BesSite>();

	/**
	 * @param _location
	 * @throws HiLAFactoryException
	 */
	private BesSite(Location _location, Object... _extraInformation)
			throws HiLAFactoryException {
		super(_location);

		for (Object object : _extraInformation) {
			if (object instanceof FactoryClient) {
				factory = (FactoryClient) object;
			}
		}
		if (factory == null) {
			throw new HiLAFactoryException(
					"Cannot instantiate BES site without BES Factory.");
		}
	}

	public static final synchronized BesSite locate(Location _location,
			Object... _extraInformation) throws HiLAException {
		if (sitesCache.containsKey(_location)) {
			return sitesCache.get(_location);
		}

		BesSite newSite = new BesSite(_location, _extraInformation);

		sitesCache.put(_location, newSite);
		return newSite;
	}

	/**
	 * @see eu.unicore.hila.grid.Site#getStorage(java.lang.String)
	 */
	@Override
	public Storage getStorage(String _storageName) throws HiLAException {
		throw new HiLANotSupportedException(
				"OGSA-BES does not support storages.");
	}

	/**
	 * @see eu.unicore.hila.grid.Submittable#submit(eu.unicore.hila.job.model.JobModel)
	 */
	@Override
	public Job submit(JobModel _jd) throws HiLAException {
		ModelToJsdl m2j = new ModelToJsdl(_jd);

		CreateActivityDocument cad = CreateActivityDocument.Factory
				.newInstance();
		cad.addNewCreateActivity().addNewActivityDocument()
				.setJobDefinition(m2j.getJSDL().getJobDefinition());

		try {
			CreateActivityResponseDocument card = factory.createActivity(cad);
			EndpointReferenceType actEpr = card.getCreateActivityResponse()
					.getActivityIdentifier();
			ActivityClient ac = new ActivityClient(actEpr.getAddress()
                    .getStringValue(), actEpr,
                    factory.getSecurityConfiguration());
			String aId = WSServerUtilities.extractResourceID(ac.getEPR());
			Location actLoc = getLocation().getChildLocation("jobs")
					.getChildLocation(aId);
			if (log.isTraceEnabled()) {
				log.trace("Activity location: " + actLoc);
			}
			return (Job) actLoc.locate(ac);
		} catch (NotAcceptingNewActivitiesFault e) {
			throw new HiLAException(
					"Cannot submit new activities to this factory.", e);
		} catch (InvalidRequestMessageFault e) {
			throw new HiLAException("Invalid request.", e);
		} catch (UnsupportedFeatureFault e) {
			throw new HiLANotSupportedException(
					"This feature is not supported.", e);
		} catch (Exception e) {
			throw new HiLAException(
					"Unable to create ActivityClient from EPR.", e);
		}
	}

	/**
	 * @see eu.unicore.hila.grid.Submittable#getTasks()
	 */
	@Override
	public List<Job> getTasks() throws HiLAException {
		throw new HiLANotSupportedException(
				"Retrieval of activities from BES factory not supported.");
	}

	/**
	 * @see eu.unicore.hila.grid.Submittable#getTask(java.lang.String)
	 */
	@Override
	public Job getTask(String _taskId) throws HiLAException {
		// TODO Auto-generated method stub
		return null;
	}

	/**
	 * @see eu.unicore.hila.grid.Reservable#reserve(eu.unicore.hila.grid.resources.ResourceDescription,
	 *      java.util.Calendar)
	 */
	@Override
	public Reservation reserve(ResourceDescription _rd, Calendar _startTime)
			throws HiLAException {
		throw new HiLANotSupportedException(
				"OGSA-BES does not support reservations.");
	}

	/**
	 * @see eu.unicore.hila.grid.Reservable#getReservations()
	 */
	@Override
	public List<Reservation> getReservations() throws HiLAException {
		throw new HiLANotSupportedException(
				"OGSA-BES does not support reservations.");
	}

	/**
	 * @see eu.unicore.hila.grid.Reservable#getReservation(java.lang.String)
	 */
	@Override
	public Reservation getReservation(String name) throws HiLAException {
		throw new HiLANotSupportedException(
				"OGSA-BES does not support reservations.");
	}

}
