/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.junit.Assert;
import org.junit.Test;
import org.unigrids.x2006.x04.services.tss.ApplicationResourceType;
import org.unigrids.x2006.x04.services.tss.TargetSystemPropertiesDocument;

import eu.unicore.hila.Location;
import eu.unicore.hila.Metadata;
import eu.unicore.hila.Resource;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLALocationSyntaxException;
import eu.unicore.hila.grid.File;
import eu.unicore.hila.grid.Job;
import eu.unicore.hila.grid.SimpleTransfer;
import eu.unicore.hila.grid.Site;
import eu.unicore.hila.grid.StatusChangeListener;
import eu.unicore.hila.grid.StatusChangedEvent;
import eu.unicore.hila.grid.Storage;
import eu.unicore.hila.grid.Task;
import eu.unicore.hila.grid.TaskStatus;
import eu.unicore.hila.grid.ThirdPartyTransfer;
import eu.unicore.hila.grid.job.JobDescription;
import eu.unicore.hila.job.jsdl.ModelToJsdl;
import eu.unicore.hila.job.model.JobModel;

/**
 * @author bjoernh
 * 
 *         13.08.2009 13:01:20
 * 
 */
public class Unicore6GridTest extends AbstractUnicore6Test {
	private static final Logger log = Logger.getLogger(Unicore6GridTest.class);

	private static final int SUBMIT_TASKS = 5;

	@Test
	public void testGetUnicore6Grid() throws HiLAException {
		Location loc = new Location("unicore6:/");

		log.info("LocationType is " + loc.getLocationType());

		Resource grid = loc.locate();
		Assert.assertNotNull("Resource could not be located.", grid);
		log.info("Location of " + loc.getLocationType().getSimpleName()
				+ " is " + grid.getLocation());

		List<Resource> gridCollections = grid.getChildren();
		for (Resource gridCollection : gridCollections) {
			log.info("ResourceType of childLocation "
					+ gridCollection.getLocation() + " is "
					+ gridCollection.getClass().getSimpleName());
			List<Resource> sites = gridCollection.getChildren();
			for (Resource site : sites) {
				log.info("Site location is " + site.getLocation());
				if (site instanceof Site) {
					List<Job> tasks = ((Site) site).getTasks();
					for (Task task : tasks) {
						log.info(task.getClass().getSimpleName() + " "
								+ task.getLocation());
					}
				}
			}
		}

	}

	@Test
	public void testSiteCachingBehavior() {
		try {
			Location loc = new Location(siteLoc);
			long time1 = System.currentTimeMillis();
			Site site = (Site) loc.locate();
			long time2 = System.currentTimeMillis();
			log.info("Initial time for lookup of site: "
					+ Long.toString(time2 - time1) + "ms.");
			log.info("Task object id: " + site);

			for (int i = 0; i < 20; i++) {
				time1 = System.currentTimeMillis();
				for (int j = 0; j < 100; j++) {
					site = (Site) loc.locate();
				}
				time2 = System.currentTimeMillis();
				log.info("Locating 100 times took "
						+ Long.toString(time2 - time1) + "ms.");
				// log.info("Task object id: " + site);
			}

		} catch (HiLAException e) {
			log.error(e);
		}
	}

	@Test
	public void testGetAllSites() throws HiLAException, InterruptedException {
		Location loc = new Location("unicore6:/sites");
		Resource res = loc.locate();
		List<Resource> sites = res.getChildren();
		for (Resource resource : sites) {
			log.info(resource.getLocation());
		}

		// once more because of caching
		long refreshInterval = Long.parseLong(((Unicore6Grid) loc
				.getParentLocation().locate()).getProperties().getProperty(
				"hila.unicore6.refresh", "60000"));
		log.info("Sleeping for " + (refreshInterval + 1000) + "ms.");
		Thread.sleep(refreshInterval + 1000);
		res.getChildren();
	}

	@Test
	public void testGetSite() throws HiLAException {
		final Location loc1 = new Location(siteLoc);
		final Resource res1 = loc1.locate();
		Assert.assertTrue("Should have located a site.", res1 instanceof Site);

		final Location loc2 = new Location(siteLoc.concat("/"));
		final Resource res2 = loc2.locate();
		Assert.assertTrue("Should have located a site.", res2 instanceof Site);
	}

	// @Test
	public void testGetTasks() throws HiLAException {
		Location loc = new Location(siteLoc);
		Resource res = loc.locate();
		Assert.assertTrue("Located resource was not a Site.",
				res instanceof Site);
		Site site = (Site) res;
		List<Job> tasks = site.getTasks();
		for (Task task : tasks) {
			log.info(task.getLocation() + " : " + task.status());
		}
	}

	@Test
	public void testSubmitAndRun() throws HiLAException {
		Location loc = new Location(siteLoc);
		Resource res = loc.locate();
		Assert.assertTrue("Located resource was not a Site.",
				res instanceof Site);
		Site site = (Site) res;
		JobModel jm = createJobModel();
		List<Task> tasks = new ArrayList<Task>();
		final long time1 = System.currentTimeMillis();
		for (int i = 0; i < SUBMIT_TASKS; i++) {
			Job task = site.submit(jm);
			task.startASync();
			tasks.add(task);
			if ((i > 0) && (i % 10) == 0) {
				final long time_int = System.currentTimeMillis();
				log.info(Long.toString((time_int - time1) / i).concat(
						"ms per task."));
			}
		}
		final long time2 = System.currentTimeMillis();
		log.info("Submission of " + Integer.toString(SUBMIT_TASKS)
				+ " tasks took " + Long.toString(time2 - time1) + "ms.");
		while (tasks.size() > 0) {
			List<Task> removables = new ArrayList<Task>();
			for (Task task : tasks) {
				log.debug("Task status of " + task.getLocation() + ": "
						+ task.status());
				TaskStatus status = task.status();
				if (status.equals(TaskStatus.ABORTED)
						|| status.equals(TaskStatus.FAILED)
						|| status.equals(TaskStatus.SUCCESSFUL)) {
					removables.add(task);
				}
			}
			// only keep monitoring those that can still change
			tasks.removeAll(removables);
		}
	}

	/**
	 * @return
	 */
	private JobModel createJobModel() {
		final JobModel jm = new JobModel();

		jm.setApplicationName("Date");
		jm.setApplicationVersion("1.0");

		return jm;
	}

	@Test
	public void testStatusChangedListener() throws HiLAException {
		log.info("testStatusChangedListener");
		JobModel jd = createJobModel();
		Site site = (Site) new Location(siteLoc).locate();

		Job ct = site.submit(jd);
		StatusChangeListener scl = new StatusChangeListener() {

			@Override
			public void statusChanged(StatusChangedEvent e) {
				log.info("Status of Task: " + e.getTask().getLocation()
						+ " changed to " + e.getStatus());
			}
		};
		ct.registerStatusChangeListener(scl);
		ct.startASync();
		try {
			Thread.sleep(10000);
		} catch (InterruptedException e1) {
			// TODO Auto-generated catch block
			log.error("CHANGE ME: bogus error message", e1);
		}

		ct.removeStatusChangeListener(scl);
	}

	public void ttestGetTask() throws HiLAException {
		Location loc = new Location(siteLoc);
		Resource res = loc.locate();

		if (res instanceof Site) {
			Site site = (Site) res;
			String[] taskIds = { "a324501b-8fe7-4d0e-8087-0bd25adcb56b" };
			for (String taskId : taskIds) {
				Task task = site.getTask(taskId);
				log.info(task.getLocation());
			}
		}
	}

	public void ttestLocateTask() throws HiLAException {
		final Location loc = new Location(
				siteLoc.concat("/tasks/a324501b-8fe7-4d0e-8087-0bd25adcb56b"));
		Resource res = loc.locate();
		if (res instanceof Task) {
			Task task = (Task) res;
			log.info(task.getLocation() + " : " + task.status());
			if (task instanceof Job) {
				Job ct = (Job) task;
				File file = ct.getWorkingDirectory();
				List<File> wdFiles = file.ls();
				for (File file2 : wdFiles) {
					log.info(file2.getLocation());
				}
			}
		}
	}

	public void ttestLocateFileInUSpace() throws HiLAException {
		Location loc = new Location(
				siteLoc.concat("/tasks/a324501b-8fe7-4d0e-8087-0bd25adcb56b/wd/files/stderr"));
		Resource res = loc.locate();
		if (res instanceof File) {
			File file = (File) res;
			System.out.println(file.getLocation() + " : "
					+ Long.toString(file.size()));
		}
	}

	// @Test
	public void testStartNEWTasks() throws HiLAException {
		Location loc = new Location(siteLoc);
		Resource res = loc.locate();

		if (res instanceof Site) {
			Site site = (Site) res;
			List<Job> tasks = site.getTasks();
			for (Job task : tasks) {
				if (task.status().equals(TaskStatus.NEW)) {
					task.startASync();
					log.info("Started job " + task.getLocation());
				}
			}
		}
	}

	@Test
	public void testTaskCleanup() throws HiLAException {
		Location loc = new Location(siteLoc);
		Resource res = loc.locate();
		if (res instanceof Site) {
			Site site = (Site) res;
			List<Job> tasks = site.getTasks();
			final long time1 = System.currentTimeMillis();
			for (Job task : tasks) {
				log.info("Cleaning up Task " + task.getLocation());
				task.cleanup();
			}
			final long time2 = System.currentTimeMillis();
			log.info("Cleaning up " + tasks.size() + " Tasks took "
					+ Long.toString(time2 - time1) + "ms.");
		}
	}

	public void testAbortTask() throws HiLAException {
		Location loc = new Location(
				"unicore6:/sites/VSGC_200909231432/tasks/e66f1c5f-4785-47a8-82f5-edae87c343bf");
		Resource res = loc.locate();
		if (res instanceof Task) {
			Task task = (Task) res;
			task.abort();
			log.info(task.getLocation() + " : " + task.status());
		}
	}

	@Test
	public void testGetStorages() throws HiLAException {
		Location loc = new Location(siteLoc);
		Resource res = loc.locate();
		if (res instanceof Site) {
			Site site = (Site) res;
			log.info(site.getLocation());
			List<Storage> storages = site.getStorages();
			for (Storage storage : storages) {
				log.info(storage.getLocation());
			}
		}
	}

	@Test
	public void testFileExists() throws HiLAException {
		final Location loc1 = new Location(siteLoc);
		final Resource res1 = loc1.locate();
		if (res1 instanceof Site) {
			Site site = (Site) res1;
			List<Storage> storages = site.getStorages();
			for (Storage storage : storages) {
				if (storage.getName().equalsIgnoreCase("home")) {
					File file = storage.asFile();
					// log.info("File emacs exists: " +
					// ((File)file.getChild(".emacs")).exists());
					log.info("File.exists() returns "
							+ Boolean.toString(file.exists()));
				}
			}
		}
		final Location loc2 = new Location(
				siteLoc.concat("/storages/Home/files/.bashrc"));
		final Resource res2 = loc2.locate();
		Assert.assertTrue("Location doesn't point to a file. Resource is a "
				+ res2.getName(), res2 instanceof File);
		File file = (File) res2;
		Assert.assertTrue(file.exists());
	}

	// @Test
	public void testListFiles() throws HiLAException {
		final Location loc1 = new Location(siteLoc);
		final Resource res1 = loc1.locate();
		if (res1 instanceof Site) {
			Site site = (Site) res1;
			List<Storage> storages = site.getStorages();
			for (Storage storage : storages) {
				if (storage.getName().equalsIgnoreCase("home")) {
					File file = storage.asFile();
					List<Resource> files = file.getChildren();
					for (Resource resource : files) {
						log.info(resource.getLocation());
					}
				}
			}
		}
	}

	@Test
	public void testExportFile() throws HiLAException {
		final Location loc = new Location(siteLoc);
		final Location fileLoc = loc
				.getChildLocation("storages/Home/files/.emacs");
		if (fileLoc.isLocationOfType(File.class)) {
			File file = (File) fileLoc.locate();
			if (file.exists()) {
				java.io.File localFile = new java.io.File("/tmp/hila_test.txt");
				Task trsf = file.exportToLocalFile(localFile);
				trsf.block();
				log.info("Export finished: " + trsf.status());
			} else {
				log.info("File " + file.getLocation() + " does not exist.");
			}
		}

	}

	@Test
	public void testListRecursive() throws HiLAException {
		final Location loc = new Location(
				siteLoc.concat("/storages/Home/files/.emacs.d"));
		File emacsD = (File) loc.locate();
		if (emacsD.isDirectory()) {
			List<File> recList = new ArrayList<File>();
			recursivelyList(emacsD, recList);
			for (File file : recList) {
				log.info(file.getLocation());
			}
		}
	}

	/**
	 * @param emacsD
	 * @param recList
	 * @throws HiLAException
	 */
	private void recursivelyList(File emacsD, List<File> recList)
			throws HiLAException {
		List<Resource> files = emacsD.getChildren();
		for (Resource resource : files) {
			if (resource instanceof File) {
				File file = (File) resource;
				recList.add(file);
				if (file.isDirectory()) {
					recursivelyList(file, recList);
				}
			}
		}
	}

	// @Test
	public void ttestTransfer() throws HiLAException {
		List<Resource> sites = new Location("unicore6:/sites").locate()
				.getChildren();
		Assert.assertTrue("Need at least two sites to test transfer.",
				sites.size() > 2);

		java.io.File localFile = new java.io.File("pom.xml");

		File firstTarget = (File) sites.get(0).getChild(
				"storages/Home/files/" + UUID.randomUUID().toString());
		File secondTarget = (File) sites.get(1).getChild(
				"storages/Home/files/" + UUID.randomUUID().toString());

		Assert.assertFalse("Target file must not exist.", firstTarget.exists());
		SimpleTransfer firstImport = firstTarget.importFromLocalFile(localFile);
		Assert.assertEquals(TaskStatus.SUCCESSFUL, firstImport.block());

		Assert.assertFalse("Target file must not exist.", secondTarget.exists());
		ThirdPartyTransfer transfer = firstTarget.transfer(secondTarget);
		Assert.assertEquals(TaskStatus.SUCCESSFUL, transfer.block());

		Assert.assertTrue(firstTarget.delete());

		java.io.File secondLocalFile = new java.io.File(UUID.randomUUID()
				.toString());
		Assert.assertFalse("local file must not exist",
				secondLocalFile.exists());

		SimpleTransfer export = secondTarget.exportToLocalFile(secondLocalFile);
		Assert.assertEquals(TaskStatus.SUCCESSFUL, export.block());

		Assert.assertTrue(secondTarget.delete());

		System.out.println(localFile.length());
		System.out.println(secondLocalFile.length());

		secondLocalFile.delete();
	}

	/*
	 * public void testDelete() throws HiLAException { final Location loc1 = new
	 * Location(siteLoc); Site site = (Site) loc1.locate(); final Location loc2
	 * = new Location(siteLoc
	 * .concat("/storages/Home/files/.emacs.d/%23test%23"));
	 * ((File)loc2.locate()).delete(); }
	 */

	@Test
	public void testGetStdout() throws HiLAException, IOException {
		JobModel jd = createJobModel();
		Site site = (Site) new Location(siteLoc).locate();
		Job t = site.submit(jd);
		TaskStatus status = t.startSync();
		System.out.println("Task finished with status " + status.toString());
		File f = t.getStdOut();
		System.out.println(f.getLocation().toString());
		java.io.File localFile = java.io.File.createTempFile("hila", null);
		localFile.delete();
		SimpleTransfer export = f.exportToLocalFile(localFile);
		System.out.println("Exporting stdout to" + localFile.getAbsolutePath());
		TaskStatus exportStatus = export.block();
		System.out.println("Export of stdout: " + exportStatus.toString());
	}

	@Test
	public void testGetMetadata() throws HiLALocationSyntaxException,
			HiLAException {
		Site site = (Site) new Location(siteLoc).locate();
		Metadata md = site.getMetadata();

		List<ApplicationResourceType> applications = (List<ApplicationResourceType>) md
				.getData("applications");
		if (applications != null) {
			log.info("# of applications: "
					+ Integer.toString(applications.size()));
			for (ApplicationResourceType application : applications) {
				log.info("Application " + application.getApplicationName()
						+ " : " + application.getApplicationVersion());
			}
		}

		TargetSystemPropertiesDocument tpd = (TargetSystemPropertiesDocument) md
				.getData("resourceProperties");
		if (tpd != null) {
			log.info("IndividualCPUCount   : "
					+ tpd.getTargetSystemProperties().getIndividualCPUCount()
							.getExactArray(0).getDoubleValue());
			log.info("IndividualCPUTime    : "
					+ tpd.getTargetSystemProperties().getIndividualCPUTime()
							.getExactArray(0).getDoubleValue());
			log.info("Individual phys. mem.: "
					+ tpd.getTargetSystemProperties()
							.getIndividualPhysicalMemory().getExactArray(0)
							.getDoubleValue());
			log.info("OS Name              : "
					+ tpd.getTargetSystemProperties().getOperatingSystem()
							.getOperatingSystemType().getOperatingSystemName()
							.toString());
			log.info("OS Version           : "
					+ tpd.getTargetSystemProperties().getOperatingSystem()
							.getOperatingSystemVersion());
			log.info("CPU/Architecture     : "
					+ tpd.getTargetSystemProperties().getProcessor()
							.getCPUArchitecture().getCPUArchitectureName()
							.toString());
			log.info("CPU/Count            : "
					+ tpd.getTargetSystemProperties().getProcessor()
							.getIndividualCPUCount().getExactArray(0)
							.getDoubleValue());
			log.info("TotalResourceCount   : "
					+ tpd.getTargetSystemProperties().getTotalResourceCount()
							.getExactArray(0).getDoubleValue());
			// CPU/Speed not available
			// log.info("CPU/Speed            : " +
			// tpd.getTargetSystemProperties().getProcessor().getIndividualCPUSpeed().getExactArray(0).getDoubleValue());
		}

	}

	/**
	 * Test submitting with the old style (JobDescription) submit method which
	 * has been deprecated @since 2.3
	 * 
	 * @throws HiLAException
	 * @throws HiLALocationSyntaxException
	 */
	@SuppressWarnings("deprecation")
	@Test
	public void testSubmitOldStyle() throws HiLALocationSyntaxException,
			HiLAException {
		final JobModel jm = createJobModel();
		((Site) new Location(siteLoc).locate()).submit(new JobDescription() {

			@Override
			public Object getActualJobDescription() {
				try {
					return new ModelToJsdl(jm).getJSDL();
				} catch (HiLAException e) {
					throw new RuntimeException(e);
				}
			}
		});
	}

}
