/*********************************************************************************
 * Copyright (c) 2010-2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Callable;

import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.uas.client.TransferControllerClient;
import de.fzj.unicore.wsrflite.xfire.ClientException;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceNotDestroyedFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnavailableFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnknownFault;
import eu.unicore.hila.HiLA;
import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseThirdPartyTransfer;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLANotImplementedException;
import eu.unicore.hila.exceptions.HiLAResourceNotFoundException;
import eu.unicore.hila.grid.Progressable;
import eu.unicore.hila.grid.TaskStatus;
import eu.unicore.hila.grid.ThirdPartyTransfer;
import eu.unicore.util.httpclient.IClientConfiguration;

/**
 * @author bjoernh
 * 
 *         17.03.2010 09:15:31
 * 
 */
@ResourceType(locationStructure = {
	"unicore6:/sites/{site}/storages/{storage}/transfers/{transfer}",
	"unicore6:/{user}@sites/{site}/storages/{storage}/transfers/{transfer}",
	"unicore6:/sites/{site}/tasks/{task}/wd/transfers/{transfer}",
	"unicore6:/{user}@sites/{site}/tasks/{task}/wd/transfers/{transfer}",
	"unicore6:/storages/{storage}/transfers/{transfer}",
	"unicore6:/{user}@storages/{storage}/transfers/{transfer}" })
public class Unicore6TransferTask extends BaseThirdPartyTransfer implements
	ThirdPartyTransfer, Callable<TaskStatus>, Progressable {

    Unicore6File source;
    Unicore6File destination;
    StorageClient sourceStorageClient;
    private StorageClient destinationStorageClient;
    private TransferControllerClient tcc;
    private long maxBytes;
    private long currentBytes;

    /**
     * @param location
     * @param sourceStorageClient
     * @throws HiLAException
     */
    public Unicore6TransferTask(Unicore6File _source,
	    StorageClient _sourceStorageClient, Unicore6File _destination,
         StorageClient _destinationStorageClient)
	    throws HiLAException {
	super(_source.getStorage().getLocation().getChildLocation("transfers")
		.getChildLocation(UUID.randomUUID().toString()));

	this.source = _source;
	this.destination = _destination;
	this.sourceStorageClient = _sourceStorageClient;

	try {
            IClientConfiguration tdSecProp = _destinationStorageClient
                    .getSecurityConfiguration().clone();
	    tdSecProp.getETDSettings().setExtendTrustDelegation(true);
	    tdSecProp.getETDSettings().setIssuerCertificateChain(
                    tdSecProp.getCredential().getCertificateChain());

	    this.destinationStorageClient = new StorageClient(
		    _destinationStorageClient.getEPR(), tdSecProp);

	} catch (CloneNotSupportedException e) {
	    throw new HiLAException("Unable to clone security properties.", e);
	} catch (Exception e) {
	    throw new HiLAException(
		    "Unable to create storage client with trust delegation.");
	}

	setCurrentState(TaskStatus.NEW);

      HiLA.getExecutor().submit(this);

	Unicore6TransfersCollection tc = (Unicore6TransfersCollection) this
		.getParent();
	if (tc != null) {
	    tc.addTask(this);
	}
    }

    public static synchronized Unicore6TransferTask locate(Location _location,
	    Object... _extraInformation) throws HiLAException {
	Unicore6TransfersCollection transfersCollection = (Unicore6TransfersCollection) _location
		.getParentLocation().locate();
	List<Resource> transfers = transfersCollection.getChildren();
	for (Resource resource : transfers) {
	    if (resource.getLocation().equals(_location)) {
		return (Unicore6TransferTask) resource;
	    }
	}
	throw new HiLAResourceNotFoundException(
		"This transfer does not exist: " + _location);
    }

    /**
     * @see java.util.concurrent.Callable#call()
     */
    public TaskStatus call() throws Exception {
	setCurrentState(TaskStatus.PENDING);

	String ftProto = "BFT";
	String sourceEPR = ftProto + ":"
		+ sourceStorageClient.getEPR().getAddress().getStringValue()
		+ "#" + source.getPath();
	tcc = destinationStorageClient.fetchFile(sourceEPR,
		destination.getPath());
	setCurrentState(TaskStatus.RUNNING);
	getCurrent();
	getMax();
	while (!tcc.isComplete()) {
	    Thread.sleep(2000);
	}
	getCurrent();
	getMax();
	if (tcc.hasFailed()) {
	    HiLAException e = new HiLAException("Remote transfer has failed.");
	    setCurrentState(TaskStatus.FAILED, e);
	    tcc.destroy();
	    tcc = null;
	    throw e;
	} else {
	    setCurrentState(TaskStatus.SUCCESSFUL);
	}
	if (tcc != null) {
	    tcc.destroy();
	    tcc = null;
	}
	return status();
    }

    /**
     * @see eu.unicore.hila.grid.Task#abort()
     */
    public void abort() throws HiLAException {
	try {
	    tcc.destroy();
	    tcc = null;
	} catch (BaseFault e) {
	    throw new HiLAException("Unable to abort transfer", e);
	} catch (ResourceUnavailableFault e) {
	    throw new HiLAException("Unable to abort transfer", e);
	} catch (ResourceUnknownFault e) {
	    throw new HiLAException("Unable to abort transfer", e);
	} catch (ResourceNotDestroyedFault e) {
	    throw new HiLAException("Unable to abort transfer", e);
	} catch (ClientException e) {
	    throw new HiLAException("Unable to abort transfer", e);
	}
    }

    /**
     * @see eu.unicore.hila.grid.Task#getId()
     */
    public String getId() {
	return this.location.getName();
    }

    /**
     * @see eu.unicore.hila.Resource#getChildren()
     */
    public List<Resource> getChildren() throws HiLAException {
	return Collections.emptyList();
    }

    // Progressable implementations
    /**
     * @see eu.unicore.hila.common.grid.BaseTask#getUnit()
     */
    public String getUnit() throws HiLANotImplementedException {
	return "bytes";
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseTask#getMax()
     */
    public synchronized long getMax() throws HiLAException {
	if (tcc != null) {
	    this.maxBytes = tcc.getSize();
	}
	return this.maxBytes;
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseTask#getCurrent()
     */
    public synchronized long getCurrent() {
	if (tcc != null) {
	    this.currentBytes = tcc.getTransferredBytes();
	}
	return this.currentBytes;
    }

}
