/*********************************************************************************
 * Copyright (c) 2010 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Callable;

import eu.unicore.hila.HiLA;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.grid.Progressable;
import eu.unicore.hila.grid.SimpleTransfer;
import eu.unicore.hila.grid.TaskStatus;

/**
 * @author bjoernh
 * 
 *         25.10.2010 13:22:36
 * 
 */
public class Unicore6RecursiveExportTask extends Unicore6ExportTask implements
	SimpleTransfer, Callable<TaskStatus>, Progressable {

    final private File destination;
    final private List<SimpleTransfer> transfers;
    final private Unicore6File source;
    private boolean abort = false;
    private boolean allCreated = false;

    /**
     * @param _location
     * @throws HiLAException
     */
    public Unicore6RecursiveExportTask(Unicore6File _source, File _destination) throws HiLAException {
	super(_source.getStorage().getLocation().getChildLocation("exports")
		.getChildLocation(UUID.randomUUID().toString()));

	this.source = _source;
	this.destination = _destination;
	this.transfers = new ArrayList<SimpleTransfer>();

	setCurrentState(TaskStatus.NEW);

	HiLA.getExecutor().submit(this);

	Unicore6ExportsCollection ec = (Unicore6ExportsCollection) this
		.getParent();
	if (ec != null) {
	    ec.addTask(this);
	}
    }

    /**
     * @see eu.unicore.hila.Task#abort()
     */
    @Override
    public void abort() throws HiLAException {
	abort = true;
	synchronized (this) {
	    if (!allCreated) {
		try {
		    wait();
		} catch (InterruptedException e) {

		}
	    }
	}
	for (SimpleTransfer transfer : transfers) {
	    transfer.abort();
	}
    }

    /**
     * @see java.util.concurrent.Callable#call()
     */
    @Override
    public TaskStatus call() throws Exception {
	setCurrentState(TaskStatus.RUNNING);

	if (!destination.exists()) {
	    destination.mkdir();
	}

	synchronized (this) {
	    List<eu.unicore.hila.grid.File> files = source.ls();
	    for (eu.unicore.hila.grid.File file : files) {
		transfers.add(file.exportToLocalFile(destination, false, true));
		if (abort) {
		    break;
		}
	    }
	    allCreated = true;
	    notify();
	}

	while (TasksHelper.hasPendingTasks(transfers)) {
	    Thread.sleep(500);
	}

	if (TasksHelper.allTasksSuccessful(transfers)) {
	    setCurrentState(TaskStatus.SUCCESSFUL);
	} else {
	    HiLAException e = new HiLAException(
		    "Not all transfers were successful.");
	    setCurrentState(TaskStatus.FAILED, e);
	    throw e;
	}

	return status();
    }

}
