/*********************************************************************************
 * Copyright (c) 2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.chemomentum.common.ws.IServiceOrchestrator;
import org.chemomentum.workassignment.xmlbeans.SubmitWorkAssignmentRequestDocument;
import org.chemomentum.workassignment.xmlbeans.WorkAssignmentDocument;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDefinitionDocument;

import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.BaseResource;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLAFactoryException;
import eu.unicore.hila.grid.Job;
import eu.unicore.hila.grid.Submittable;
import eu.unicore.hila.grid.job.JobDescription;
import eu.unicore.hila.job.model.JobModel;
import eu.unicore.hila.job.model.MappingFactory;
import eu.unicore.hila.job.spi.MappingFactorySpi;
import eu.unicore.hila.job.spi.ModelToNative;
import eu.unicore.hila.job.spi.NativeToModel;
import eu.unicore.hila.job.spi.NoSuchMappingException;

/**
 * @author bjoernh
 * 
 *         13.01.2011 08:44:21
 * 
 */
@ResourceType(locationStructure = { "unicore6:/orchestrators/{orchestrator}/?",
		"unicore6:/{user}@orchestrators/{orchestrator}/?" })
public class Unicore6Orchestrator extends BaseResource implements Submittable {

	private static Map<Location, Unicore6Orchestrator> orchestrators = new HashMap<Location, Unicore6Orchestrator>();

	private IServiceOrchestrator orchestrator;

	private Unicore6SecurityProperties secprop;

	private List<Job> tasks;

	/**
	 * @param _location
	 */
	private Unicore6Orchestrator(Location _location,
			IServiceOrchestrator _orchestrator,
			Unicore6SecurityProperties _secprop) {
		super(_location);

		this.tasks = new ArrayList<Job>();

		this.orchestrator = _orchestrator;
		this.secprop = _secprop;
	}

	public static synchronized Unicore6Orchestrator locate(Location _location,
			Object... _extraInformation) throws HiLAException {
		if (orchestrators.containsKey(_location)) {
			return orchestrators.get(_location);
		}

		IServiceOrchestrator orchestrator = null;
		Unicore6SecurityProperties secprop = null;
		for (Object object : _extraInformation) {

			if (object instanceof IServiceOrchestrator) {
				orchestrator = (IServiceOrchestrator) object;
			} else if (object instanceof Unicore6SecurityProperties) {
				secprop = (Unicore6SecurityProperties) object;
			}

		}

		if ((orchestrator != null) && (secprop != null)) {
			Unicore6Orchestrator u6orchestrator = new Unicore6Orchestrator(
					_location, orchestrator, secprop);
			orchestrators.put(_location, u6orchestrator);
			return u6orchestrator;
		}

		_location.getParentLocation().locate().getChildren();
		return (Unicore6Orchestrator) _location.locate();

	}

	/**
	 * @see eu.unicore.hila.Resource#getChildren()
	 */
	@Override
	public List<Resource> getChildren() throws HiLAException {
		List<Resource> resources = new ArrayList<Resource>();
		resources.addAll(getTasks());
		return resources;
	}

	/**
	 * @see eu.unicore.hila.grid.Submittable#submit(eu.unicore.hila.grid.job.JobDescription)
	 */
	@Override
   public Job submit(JobModel _jd) throws HiLAException {
      MappingFactorySpi mapping;
      try {
         mapping = MappingFactory.getMapping("jsdl");
      } catch (NoSuchMappingException e) {
         throw new HiLAException("Unable to map job model to JSDL.", e);
      }
      ModelToNative m2n = mapping.createModelToNativeMapping(_jd);
      JobDefinitionDocument actualJobDescription = (JobDefinitionDocument) m2n
            .getNative();

      WorkAssignmentDocument theWA = null;

      if (actualJobDescription instanceof JobDefinitionDocument) {
         JobDefinitionDocument jsdl = (JobDefinitionDocument) actualJobDescription;
         // wrap it inside a work assignment
         WorkAssignmentDocument wad = WorkAssignmentDocument.Factory
               .newInstance();
         wad.addNewWorkAssignment().addNewWork()
               .setJobDefinition(jsdl.getJobDefinition());
         // set id
         wad.getWorkAssignment().setId(UUID.randomUUID().toString());
         theWA = wad;
      } else if (actualJobDescription instanceof WorkAssignmentDocument) {
         theWA = (WorkAssignmentDocument) actualJobDescription;
      } else {
         throw new HiLAException("I don't understand this job description.");
      }

      SubmitWorkAssignmentRequestDocument swareqd = SubmitWorkAssignmentRequestDocument.Factory
            .newInstance();
      swareqd.addNewSubmitWorkAssignmentRequest().setWorkAssignment(
            theWA.getWorkAssignment());

      orchestrator.submitWorkAssignment(swareqd);

      Unicore6OrchestratorJob orchJob = new Unicore6OrchestratorJob(
            this.location.getChildLocation(theWA.getWorkAssignment().getId()),
            orchestrator, secprop);

      tasks.add(orchJob);

      return orchJob;
   }

   /**
    * @see eu.unicore.hila.grid.Submittable#getTasks()
    */
   @Override
   public List<Job> getTasks() throws HiLAException {
      return tasks;
   }

   /**
    * @see eu.unicore.hila.grid.Submittable#getTask(java.lang.String)
    */
   @Override
   public Job getTask(String _taskId) throws HiLAException {
      // TODO Auto-generated method stub
      return null;
   }

   /**
    * @see eu.unicore.hila.grid.Submittable#submit(eu.unicore.hila.grid.job.JobDescription)
    */
   @Override
   public Job submit(JobDescription _jd) throws HiLAException {
      if (_jd.getActualJobDescription() instanceof JobDefinitionDocument) {
         JobDefinitionDocument jsdl = (JobDefinitionDocument) _jd
               .getActualJobDescription();
         MappingFactorySpi mapping;
         try {
            mapping = MappingFactory.getMapping("jsdl");
            NativeToModel n2m = mapping.createNativeToModelMapping(jsdl);
            return submit(n2m.getModel());
         } catch (NoSuchMappingException e) {
            throw new HiLAException("Cannot map jsdl to abstract model.", e);
         }
      }
      throw new HiLAFactoryException(
            "Unable to map input JobDescription to the job model.");
   }

}
