/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.shell.commands;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.ParseException;
import org.apache.log4j.Logger;

import eu.unicore.hila.Location;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.grid.Job;
import eu.unicore.hila.grid.File;
import eu.unicore.hila.shell.ShellContext;

/**
 * @author bjoernh
 * 
 *         16.10.2009 15:54:11
 * 
 */
public class Cleanup extends ShellCommandBase implements ShellCommand {

    private static final Logger log = Logger.getLogger(Cleanup.class);

    /**
     * 
     */
    public Cleanup() {
	initializeOptions();
    }

    /**
     * @see eu.unicore.hila.shell.commands.ShellCommandBase#execute(java.lang.String[],
     *      eu.unicore.hila.shell.ShellContext)
     */
    @Override
    public void execute(String[] cmdArgs, ShellContext ctx)
	    throws HiLAException {
	try {
	    CommandLine cmdLine = parser.parse(options, cmdArgs);

	    if (cmdLine.hasOption(OptionConstants.HELP.getOpt())) {
		printHelp(ctx);
		return;
	    }

	    List<String> taskLocations = cmdLine.getArgList();
	    taskLocations.remove(0); // remove command itself
	    final List<Job> tasks = new ArrayList<Job>();
	    for (String locationString : taskLocations) {
		Location taskLocation = new Location(locationString);

		if (!taskLocation.isLocationOfType(Job.class)) {
		    taskLocation = ctx.getCurrentLocation().getChildLocation(
			    locationString);
		}

		if (taskLocation.isLocationOfType(Job.class)) {
		    tasks.add((Job) taskLocation.locate());
		} else {
		    try {
			ctx
				.getCs()
				.printString(
					locationString
						+ " is not a location of type ComputeTask.");
			ctx.getCs().printNewline();
		    } catch (IOException e) {
			log.warn(locationString
				+ " is not a location of type ComputeTask.");
		    }
		}
	    }

	    for (Job computeTask : tasks) {
		List<File> exportFiles;
		if (cmdLine.hasOption(OptionConstants.EXPORT_ALL.getOpt())) {
		    if (log.isDebugEnabled()) {
			log
				.debug("Exporting all files from working directory of "
					+ computeTask.getLocation());
		    }
		    exportFiles = computeTask.getWorkingDirectory().ls();
		} else {
		    exportFiles = new ArrayList<File>();
		    String[] exports = cmdLine
			    .getOptionValues(OptionConstants.EXPORT
				    .getOpt());
		    if (exports != null) {
			for (String export : exports) {
			    if (log.isDebugEnabled()) {
				log.debug("Exporting " + export
					+ " from working directory of "
					+ computeTask.getLocation());
			    }
			    exportFiles.add((File) computeTask
				    .getWorkingDirectory().getChild(export));
			}
		    }
		}
		// cleanup should run concurrently
		computeTask.cleanup((File[]) exportFiles.toArray(new File[0]));
	    }
	} catch (ParseException e) {
	    log.error("Unable to parse command line.", e);
	}
    }

    /**
     * 
     */
    private void initializeOptions() {
	options.addOption(OptionConstants.HELP);
	options.addOption(OptionConstants.EXPORT);
	options.addOption(OptionConstants.EXPORT_ALL);
    }

}
