/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.shell;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jline.Completor;
import jline.ConsoleReader;
import jline.FileNameCompletor;
import jline.MultiCompletor;
import jline.SimpleCompletor;

import org.apache.log4j.Logger;

import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLALocationSyntaxException;
import eu.unicore.hila.shell.commands.Abort;
import eu.unicore.hila.shell.commands.Alias;
import eu.unicore.hila.shell.commands.Cat;
import eu.unicore.hila.shell.commands.Cd;
import eu.unicore.hila.shell.commands.ChMod;
import eu.unicore.hila.shell.commands.Cleanup;
import eu.unicore.hila.shell.commands.Cp;
import eu.unicore.hila.shell.commands.Exit;
import eu.unicore.hila.shell.commands.Export;
import eu.unicore.hila.shell.commands.GetMetadata;
import eu.unicore.hila.shell.commands.Help;
import eu.unicore.hila.shell.commands.Import;
import eu.unicore.hila.shell.commands.LCd;
import eu.unicore.hila.shell.commands.LLs;
import eu.unicore.hila.shell.commands.LPwd;
import eu.unicore.hila.shell.commands.Ls;
import eu.unicore.hila.shell.commands.MkDir;
import eu.unicore.hila.shell.commands.Monitor;
import eu.unicore.hila.shell.commands.Mv;
import eu.unicore.hila.shell.commands.Pwd;
import eu.unicore.hila.shell.commands.Rm;
import eu.unicore.hila.shell.commands.Run;
import eu.unicore.hila.shell.commands.ShellCommand;
import eu.unicore.hila.shell.commands.Start;
import eu.unicore.hila.shell.commands.Status;
import eu.unicore.hila.shell.commands.Submit;
import eu.unicore.hila.shell.commands.Time;
import eu.unicore.hila.shell.commands.Transfer;
import eu.unicore.hila.shell.commands.Unalias;
import eu.unicore.hila.shell.commands.Version;
import eu.unicore.hila.shell.completors.CommandMultiplexedCompletor;
import eu.unicore.hila.shell.completors.LocationCompletor;

/**
 * @author bjoernh
 * 
 *         07.10.2009 12:25:58
 * 
 */
public class HiLAShell {

    private static final Logger log = Logger.getLogger(HiLAShell.class);

    /**
     * @param args
     * @throws HiLALocationSyntaxException
     */
    public static void main(String[] args) throws HiLALocationSyntaxException {

	ConsoleReader cs = null;

	try {
	    cs = new ConsoleReader();
	    // cs.setDefaultPrompt("hila > ");
	    File hila2 = new File(System.getProperty("user.home")
		    + File.separator + ".hila2");
	    if (!hila2.exists()) {
		hila2.mkdirs();
	    } else {
		if (!hila2.isDirectory()) {
		    log.warn("Won't be able to store history, as ~/.hila2 "
			    + "exists, but is an ordinary file.");
		} else {
		    File history = new File(hila2, "shell-history");
		    cs.getHistory().setHistoryFile(history);
		}
	    }
	} catch (IOException e) {
	    log.error("Unable to open console. Cannot use HiLA Shell.", e);
	    return;
	}

	ShellContext sc = new ShellContext(cs);

	ShellCommand ls = new Ls();
	sc.registerCommand(ls.getName(), ls);

	ShellCommand exit = new Exit();
	sc.registerCommand(exit.getName(), exit);
	// an alias ;)
	sc.registerCommand("quit", exit);

	ShellCommand cd = new Cd();
	sc.registerCommand(cd.getName(), cd);
	sc.registerCommand("chdir", cd);

	ShellCommand _cleanup = new Cleanup();
	sc.registerCommand(_cleanup);

	ShellCommand _mkdir = new MkDir();
	sc.registerCommand(_mkdir);

	ShellCommand pwd = new Pwd();
	sc.registerCommand(pwd);

	ShellCommand submit = new Submit();
	sc.registerCommand(submit);

	ShellCommand help = new Help();
	sc.registerCommand(help);

	ShellCommand _export = new Export();
	sc.registerCommand(_export);

	ShellCommand _import = new Import();
	sc.registerCommand(_import);

	ShellCommand _rm = new Rm();
	sc.registerCommand(_rm.getName(), _rm);

	ShellCommand _lpwd = new LPwd();
	sc.registerCommand(_lpwd);

	ShellCommand _lcd = new LCd();
	sc.registerCommand(_lcd);

	ShellCommand _lls = new LLs();
	sc.registerCommand(_lls);

	ShellCommand _chmod = new ChMod();
	sc.registerCommand(_chmod);

	ShellCommand _start = new Start();
	sc.registerCommand(_start);

	ShellCommand _time = new Time();
	sc.registerCommand(_time);

	ShellCommand _transfer = new Transfer();
	sc.registerCommand(_transfer);

	ShellCommand _move = new Mv();
	sc.registerCommand(_move);
	sc.registerCommand("move", _move);

	ShellCommand _copy = new Cp();
	sc.registerCommand(_copy);
	sc.registerCommand("copy", _copy);

	ShellCommand _abort = new Abort();
	sc.registerCommand(_abort);
	sc.registerCommand("cancel", _abort);

	ShellCommand _monitor = new Monitor();
	sc.registerCommand(_monitor);

	ShellCommand _run = new Run();
	sc.registerCommand(_run);

	ShellCommand _status = new Status();
	sc.registerCommand(_status);

	ShellCommand _version = new Version();
	sc.registerCommand(_version);

	ShellCommand _cat = new Cat();
	sc.registerCommand(_cat);

	ShellCommand _getMetadata = new GetMetadata();
	sc.registerCommand(_getMetadata);

	final ShellCommand _alias = new Alias();
	sc.registerCommand(_alias);

	final ShellCommand _unalias = new Unalias();
	sc.registerCommand(_unalias);

	Completor argCmpl = new MultiCompletor(new Completor[] {
		new SimpleCompletor(sc.getCommands().toArray(
			new String[sc.getCommands().size()])),
		new MultiCompletor(new Completor[] {
			new CommandMultiplexedCompletor(sc),
			new LocationCompletor(sc), new FileNameCompletor() }) });

	cs.addCompletor(argCmpl);

	int ioErrorCount = 0;
	while (!sc.isDone()) {
	    try {
		String cmd = null;
		if (sc.getCurrentLocation() != null) {
		    cmd = cs.readLine(sc.getCurrentLocation().getName()
			    .toString()
			    + " :> ");
		} else {
		    cmd = cs.readLine("hila > ");
		}
		if (cmd == null) {
		    cs.printNewline();
		    cs.getHistory().flushBuffer();
		    System.exit(0);
		    return;
		}
		ioErrorCount = 0; // reset error count

		cmd = cmd.trim();
		String[] cmdArgs = splitArguments(cmd);

		if (cmdArgs.length > 0) {
		    if (sc.isAlias(cmdArgs[0])) {
			String aliasCommand = sc.getAlias(cmdArgs[0]);
			StringBuilder sb = new StringBuilder(aliasCommand);
			for (int i = 1; i < cmdArgs.length; i++) {
			    sb.append(' ' + cmdArgs[i]);
			}
			if (log.isDebugEnabled()) {
			    log.debug("Rewritten, aliased command: "
				    + sb.toString());
			}
			cmdArgs = splitArguments(sb.toString());
		    }

		    if (cmdArgs.length > 0) {

			ShellCommand exCmd = sc.getCommand(cmdArgs[0]);
			if (exCmd != null) {
			    try {
				exCmd.execute(cmdArgs, sc);
			    } catch (HiLAException e) {
				log.error(e);
			    }
			} else {
			    log.warn("Unknown command: " + cmdArgs[0]);
			}
		    }
		}
	    } catch (IOException e) {
		log.error("Cannot read from or write to console.", e);
		if (++ioErrorCount > 3) {
		    log.error("Too many IOExceptions. Exiting.");
		    return;
		}
	    }
	}

    }

    /**
     * alias ll='ls -l' should return { alias, ll='ls -l' }
     * 
     * @param cmd
     * @return
     */
    private static String[] splitArguments(String cmd) {
	ArrayList<String> arguments = new ArrayList<String>();
	Pattern p = Pattern.compile("[^\\s\"']+|\"([^\"]*)\"|'([^']*)'");
	Matcher m = p.matcher(cmd);
	while (m.find()) {
	    if (m.group(1) != null) {
		arguments.add(m.group(1));
	    } else if (m.group(2) != null) {
		arguments.add(m.group(2));
	    } else {
		arguments.add(m.group());
	    }
	}
	return arguments.toArray(new String[arguments.size()]);
    }

}
