/*********************************************************************************
 * Copyright (c) 2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.job.jsdl;

import java.util.Map;

import org.apache.xmlbeans.XmlCursor;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.ApplicationType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.DataStagingType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDefinitionDocument;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDefinitionType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDescriptionType;
import org.ggf.schemas.jsdl.x2005.x11.jsdlPosix.ArgumentType;
import org.ggf.schemas.jsdl.x2005.x11.jsdlPosix.EnvironmentType;
import org.ggf.schemas.jsdl.x2005.x11.jsdlPosix.FileNameType;
import org.ggf.schemas.jsdl.x2005.x11.jsdlPosix.POSIXApplicationDocument;
import org.ggf.schemas.jsdl.x2005.x11.jsdlPosix.POSIXApplicationType;

import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.job.model.Export;
import eu.unicore.hila.job.model.Import;
import eu.unicore.hila.job.model.JobModel;
import eu.unicore.hila.job.model.StageIn;
import eu.unicore.hila.job.model.StageOut;
import eu.unicore.hila.job.spi.ModelToNative;

/**
 * This class maps the {@link JobModel} to a JSDL document.
 * 
 * @author bjoernh
 * 
 *         22.06.2011 14:58:18
 * 
 */
public class ModelToJsdl implements ModelToNative {
    /**
    * 
    */
   protected static final String FORMAT_JSDL = "jsdl";

   private JobModel jobModel = null;

   private long lastJsdlUpdate = 0;

    private JobDefinitionDocument jsdl;

   /**
    * 
    */
   public ModelToJsdl(JobModel _model) {
      this.jobModel = _model;
   }

    public JobDefinitionDocument getJSDL() throws HiLAException {
	if ((jsdl == null) || (lastJsdlUpdate < jobModel.getLastModified())) {
         createJSDL();
	}
	return this.jsdl;
    }

    /**
     * @throws HiLAException
     * 
     */
    private void createJSDL() throws HiLAException {
	jsdl = JobDefinitionDocument.Factory.newInstance();

	JobDefinitionType jd = jsdl.addNewJobDefinition();
	JobDescriptionType jDescr = jd.addNewJobDescription();

	POSIXApplicationDocument posixAppD = POSIXApplicationDocument.Factory
		.newInstance();
	POSIXApplicationType posixApp = null;

	// application or executable
	if ((jobModel.getApplicationName() == null)
		^ (jobModel.getExecutable() == null)) {
	    ApplicationType app = jDescr.addNewApplication();
	    if (jobModel.getApplicationName() != null) {
		app.setApplicationName(jobModel.getApplicationName());
		if (jobModel.getApplicationVersion() != null) {
		    app.setApplicationVersion(jobModel.getApplicationVersion());
		}
	    } else {
		posixApp = posixAppD.addNewPOSIXApplication();
		posixApp.addNewExecutable().setStringValue(
			jobModel.getExecutable());
	    }

	    // arguments
	    if (jobModel.getArguments() != null) {
		for (String argument : jobModel.getArguments()) {
		    if (posixApp == null) {
			posixApp = posixAppD.addNewPOSIXApplication();
		    }
		    ArgumentType aType = posixApp.addNewArgument();
		    aType.setStringValue(argument);
		}
	    }
	    
	    // input, output, error
	    if(jobModel.getStdin() != null) {
	    	FileNameType inName = FileNameType.Factory.newInstance();
	    	inName.setStringValue(jobModel.getStdin());
				if (posixApp == null) {
					posixApp = posixAppD.addNewPOSIXApplication();
				}
				posixApp.setInput(inName);
	    }
			if (jobModel.getStdout() != null) {
				FileNameType outName = FileNameType.Factory.newInstance();
				outName.setStringValue(jobModel.getStdout());
				if (posixApp == null) {
					posixApp = posixAppD.addNewPOSIXApplication();
				}
				posixApp.setOutput(outName);
			}
			if (jobModel.getStderr() != null) {
				FileNameType errName = FileNameType.Factory.newInstance();
				errName.setStringValue(jobModel.getStderr());
				if (posixApp == null) {
					posixApp = posixAppD.addNewPOSIXApplication();
				}
				posixApp.setError(errName);
			}

	    // environment
	    if (jobModel.getEnvironment() != null) {
		for (Map.Entry<String, String> env : jobModel.getEnvironment()
			.entrySet()) {
		    if (posixApp == null) {
			posixApp = posixAppD.addNewPOSIXApplication();
		    }
		    EnvironmentType eType = posixApp.addNewEnvironment();
		    eType.setName(env.getKey());
		    eType.setStringValue(env.getValue());
		}
	    }

	    if (posixApp != null) {
		XmlCursor cursor = app.newCursor();
		cursor.toEndToken();
		posixApp.newCursor().copyXml(cursor);
	    }
	} else {
	    throw new HiLAException(
		    "Need exactly one application for execution.");
	}

	if (jobModel.getTaskName() != null) {
	    jDescr.addNewJobIdentification().setJobName(jobModel.getTaskName());
	}

	// data staging
	if (jobModel.getImports() != null) {
	    for (Import imprt : jobModel.getImports()) {
		DataStagingType dStag = jDescr.addNewDataStaging();
		dStag.addNewSource().setURI(imprt.getSource().toString());
		dStag.setFileName(imprt.getTargetName());
	    }
	}
	if (jobModel.getExports() != null) {
	    for (Export export : jobModel.getExports()) {
		DataStagingType dStag = jDescr.addNewDataStaging();
		dStag.addNewTarget().setURI(export.getTargetFile().toString());
		dStag.setFileName(export.getSourcePath());
	    }
	}

		if (jobModel.getStageIns() != null) {
			for (StageIn stageIn : jobModel.getStageIns()) {
				DataStagingType dStag = jDescr.addNewDataStaging();
				dStag.addNewSource().setURI(stageIn.getSourceURI());
				dStag.setFileName(stageIn.getFileName());
				dStag.setFilesystemName(stageIn.getFileSystemName());
				dStag.setDeleteOnTermination(stageIn.isDeleteOnTermination());
			}
		}

		if (jobModel.getStageOuts() != null) {
			for (StageOut stageOut : jobModel.getStageOuts()) {
				DataStagingType dStag = jDescr.addNewDataStaging();
				dStag.addNewTarget().setURI(stageOut.getTargetURI());
				dStag.setFileName(stageOut.getFileName());
				dStag.setFilesystemName(stageOut.getFileSystemName());
				dStag.setDeleteOnTermination(stageOut.isDeleteOnTermination());
			}
		}
	// end data staging

	if (jobModel.getResources() != null) {
	    jDescr.setResources(jobModel.getResources());
	}

      lastJsdlUpdate = System.currentTimeMillis();
    }

   /**
    * @throws HiLAException
    * @see eu.unicore.hila.job.spi.ModelToNative#getNative()
    */
   @Override
   public Object getNative() throws HiLAException {
      return getJSDL();
   }

   /**
    * @see eu.unicore.hila.job.spi.ModelToNative#getFormatName()
    */
   public String getFormatName() {
      return FORMAT_JSDL;
   }
}
