/*********************************************************************************
 * Copyright (c) 2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.emi.es;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.StringTokenizer;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;

import org.apache.log4j.Logger;

import eu.emi.es.x2010.x12.adl.ActivityDescriptionDocument;
import eu.emi.es.x2010.x12.creation.CreateActivityResponseDocument;
import eu.unicore.emi.es.clients.ActivityInfoClient;
import eu.unicore.emi.es.clients.ActivityManagementClient;
import eu.unicore.emi.es.clients.CreateActivityClient;
import eu.unicore.emi.es.faults.AccessControlFault;
import eu.unicore.emi.es.faults.InternalBaseFault;
import eu.unicore.emi.es.faults.InvalidActivityLimitException;
import eu.unicore.emi.es.faults.InvalidParameterException;
import eu.unicore.emi.es.faults.VectorLimitExceededFault;
import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseSite;
import eu.unicore.hila.exceptions.HiLAAccessDeniedException;
import eu.unicore.hila.exceptions.HiLACannotContactSiteException;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLANotImplementedException;
import eu.unicore.hila.exceptions.HiLANotSupportedException;
import eu.unicore.hila.exceptions.HiLAResourceNotFoundException;
import eu.unicore.hila.grid.Job;
import eu.unicore.hila.grid.Reservation;
import eu.unicore.hila.grid.Storage;
import eu.unicore.hila.grid.resources.ResourceDescription;
import eu.unicore.hila.job.emi.AdlMappingFactory;
import eu.unicore.hila.job.model.JobModel;
import eu.unicore.hila.job.model.MappingFactory;
import eu.unicore.hila.job.spi.NoSuchMappingException;
import eu.unicore.util.httpclient.IClientConfiguration;

/**
 * @author bjoernh
 * 
 *         21.12.2011 10:20:35
 * 
 */
@ResourceType(locationStructure = {
      EmiEsImplementationProvider.SCHEME + ":/sites/{siteName}/?",
      EmiEsImplementationProvider.SCHEME + ":/{user}@sites/{siteName}/?" })
public class EmiEsSite extends BaseSite {
   private static final Logger log = Logger.getLogger(EmiEsSite.class);

   private ActivityInfoClient actInfoClient = null;
   private ActivityManagementClient actMgmtClient = null;
   private CreateActivityClient creationClient = null;

   private static final Cache sitesCache;
   static {
      final CacheManager manager = CacheManager.getInstance();
      manager.addCache(new Cache(EmiEsSite.class.getName(), 500, true, true, 0,
            0));
      sitesCache = manager.getCache(EmiEsSite.class.getName());
   }

   /**
    * @param _location
    * @throws HiLACannotContactSiteException
    */
   private EmiEsSite(Location _location, Object... _extraInformation)
         throws HiLACannotContactSiteException {
      super(_location);
      for (Object object : _extraInformation) {
         if (object instanceof ActivityInfoClient) {
            actInfoClient = (ActivityInfoClient) object;
         } else if (object instanceof ActivityManagementClient) {
            actMgmtClient = (ActivityManagementClient) object;
         } else if (object instanceof CreateActivityClient) {
            creationClient = (CreateActivityClient) object;
         }
      }

      String serviceBase = null;
        IClientConfiguration securityProperties = null;
      if (actMgmtClient != null) {
         serviceBase = getServiceBase(actMgmtClient.getURL());
         securityProperties = actMgmtClient.getSecurityProperties();
      } else if (actInfoClient != null) {
         serviceBase = getServiceBase(actInfoClient.getURL());
         securityProperties = actInfoClient.getSecurityProperties();
      } else if (creationClient != null) {
         serviceBase = getServiceBase(creationClient.getUrl());
         securityProperties = creationClient.getSecurityProperties();
      }

      try {
         if (serviceBase != null) {
            if (actMgmtClient == null) {
               actMgmtClient = new ActivityManagementClient(
                     serviceBase.concat(ActivityManagementClient.SERVICE_NAME),
                     securityProperties);
            }
            if (actInfoClient == null) {
               actInfoClient = new ActivityInfoClient(
                     serviceBase.concat(ActivityInfoClient.SERVICE_NAME),
                     securityProperties);
            }
            if (creationClient == null) {
               creationClient = new CreateActivityClient(
                     serviceBase.concat(CreateActivityClient.SERVICE_NAME),
                     securityProperties);
            }
         }
      }
      catch (Exception e) {
         // this is ok, trust me!
         log.error("Error: " + e.getMessage(), e);
      }


      if (this.actInfoClient == null || this.actMgmtClient == null
            || this.creationClient == null) {
         throw new HiLACannotContactSiteException("Missing service clients.");
      }
        if (log.isDebugEnabled()) {
            log.debug("Site initialized with: " + actInfoClient.getURL() + " "
                    + actMgmtClient.getURL() + " " + creationClient.getUrl());
        }
   }

   /**
    * @param url
    * @return
    */
   private String getServiceBase(String _serviceURL) {
      StringTokenizer tok = new StringTokenizer(_serviceURL, "/");
      List<String> tokens = new ArrayList<String>();
      while (tok.hasMoreTokens()) {
         tokens.add(tok.nextToken());
      }
      List<String> baseTokens = tokens.subList(0, tokens.size() - 2);
      StringBuilder sb = new StringBuilder();
      for (String baseTok : baseTokens) {
         sb.append(baseTok + "/");
      }

      return sb.toString();
   }

   public static final EmiEsSite locate(Location _location,
         Object... _extraInformation) throws HiLAException {
      Element cacheElement = sitesCache.get(_location);
      if (cacheElement != null) {
         if (log.isDebugEnabled()) {
            log.debug("Found cache entry for " + _location);
         }
         EmiEsSite site = (EmiEsSite) cacheElement.getObjectValue();
         if (site.ok()) {
            return site;
         } else {
            sitesCache.removeElement(cacheElement);
         }
      }

      if (log.isDebugEnabled()) {
         log.debug("Cache for " + _location + " was invalid.");
      }

      if (_extraInformation.length > 0) {
         if (log.isDebugEnabled()) {
            log.debug("Creating new Site object for " + _location);
            if (log.isTraceEnabled()) {
               for (Object object : _extraInformation) {
                  log.trace("Using extraInformation " + object);
               }
            }
         }
         final EmiEsSite site = new EmiEsSite(_location, _extraInformation);
         sitesCache.put(new Element(_location, site));
         return site;
      }

      Location sitesLoc = findParentLocationOfType(EmiEsSitesCollection.class,
            _location, EmiEsGrid.class);
      for (Resource res : sitesLoc.locate().getChildren()) {
         if (res.getLocation().equals(_location)) {
            return (EmiEsSite) res;
         }
      }

      throw new HiLAResourceNotFoundException("Site " + _location
            + " cannot be found.");
   }

   /**
    * @see eu.unicore.hila.grid.Site#getStorage(java.lang.String)
    */
   @Override
   public Storage getStorage(String _storageName) throws HiLAException {
      throw new HiLANotImplementedException(
            "Site storages not available in EMI-ES.");
   }

   /**
    * @see eu.unicore.hila.grid.Submittable#submit(eu.unicore.hila.job.model.JobModel)
    */
   @Override
   public Job submit(JobModel _jd) throws HiLAException {
      try {
         ActivityDescriptionDocument ad = (ActivityDescriptionDocument) MappingFactory
               .getMapping(AdlMappingFactory.EMI_ADL_MAPPING)
               .createModelToNativeMapping(_jd).getNative();
			CreateActivityResponseDocument card = creationClient
					.createActivities(ad.getActivityDescription());
			if (card.getCreateActivityResponse()
               .getActivityCreationResponseArray().length == 1) {
				String aid = card.getCreateActivityResponse()
                  .getActivityCreationResponseArray(0).getActivityID();
            Location actLoc = getLocation().getChildLocation("tasks")
                  .getChildLocation(aid);
            return new EmiEsJob(actLoc, actInfoClient,
                  actMgmtClient);
         } else {
            throw new HiLAException("Invalid response on submit.");
         }
      } catch (NoSuchMappingException e) {
         throw new HiLAException(
               "Unable to find mapping for JobModel to EMI ActivityDescription.",
               e);
      } catch (Exception e) {
         throw new HiLAException("Error submitting job.", e);
      }
   }

   /**
    * @see eu.unicore.hila.grid.Submittable#getTasks()
    */
   @Override
   public List<Job> getTasks() throws HiLAException {
      final List<Job> jobs = new ArrayList<Job>();
      try {
         List<String> actIds = actInfoClient.listActivities();
         for (String id : actIds) {
            Location jobLocation = getLocation().getChildLocation("tasks")
                  .getChildLocation(id);
            jobs.add((Job) jobLocation.locate(actInfoClient, actMgmtClient,
                  creationClient));
         }

      } catch (InternalBaseFault e) {
         // TODO Auto-generated catch block
         log.error("CHANGE ME: bogus error message", e);
      } catch (InvalidActivityLimitException e) {
         // TODO Auto-generated catch block
         log.error("CHANGE ME: bogus error message", e);
      } catch (InvalidParameterException e) {
         // TODO Auto-generated catch block
         log.error("CHANGE ME: bogus error message", e);
      } catch (AccessControlFault e) {
         // TODO Auto-generated catch block
         log.error("CHANGE ME: bogus error message", e);
      } catch (VectorLimitExceededFault e) {
         // TODO Auto-generated catch block
         log.error("CHANGE ME: bogus error message", e);
      }
      return jobs;
   }

   /**
    * @see eu.unicore.hila.grid.Submittable#getTask(java.lang.String)
    */
   @Override
   public Job getTask(String _taskId) throws HiLAException {
      try {
         actMgmtClient.getActivityStatus(_taskId);
         // activity exists, if no exception has been thrown, so
         // we continue crafting the Job
         Location jobLocation = getLocation().getChildLocation("tasks")
               .getChildLocation(_taskId);
         return (Job) jobLocation.locate(actMgmtClient);
      } catch (InternalBaseFault e) {
         throw new HiLAException("", e);
      } catch (VectorLimitExceededFault e) {
         throw new HiLAException("", e);
      } catch (AccessControlFault e) {
         // TODO: extend HiLAAccessDeniedException to accept causes
         throw new HiLAAccessDeniedException("Cannot access activity "
               + _taskId + ": " + e.getMessage());
      }
   }

   /**
    * @see eu.unicore.hila.grid.Reservable#reserve(eu.unicore.hila.grid.resources.ResourceDescription,
    *      java.util.Calendar)
    */
   @Override
   public Reservation reserve(ResourceDescription _rd, Calendar _startTime)
         throws HiLAException {
      throw new HiLANotSupportedException(
            "EMI ES implementation does not support reservations.");
   }

   /**
    * @see eu.unicore.hila.grid.Reservable#getReservations()
    */
   @Override
   public List<Reservation> getReservations() throws HiLAException {
      throw new HiLANotSupportedException(
            "EMI ES implementation does not support reservations.");
   }

   /**
    * @see eu.unicore.hila.grid.Reservable#getReservation(java.lang.String)
    */
   @Override
   public Reservation getReservation(String name) throws HiLAException {
      throw new HiLANotSupportedException(
            "EMI ES implementation does not support reservations.");
   }

}
