/*********************************************************************************
 * Copyright (c) 2010 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.common;

import java.util.ArrayList;
import java.util.List;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.internal.runners.statements.Fail;

import eu.unicore.hila.Location;
import eu.unicore.hila.Metadata;
import eu.unicore.hila.Resource;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLALocationSyntaxException;

/**
 * @author bjoernh
 * 
 *         14.07.2010 16:43:37
 * 
 */
public class BaseResourceTest {

	private static final String RES_LOCATION = "scheme:/resource/path";
	Resource res = null;

	@Before
	public void setup() {
		try {
			res = new BaseResource(new Location(RES_LOCATION)) {

				@Override
				public List<Resource> getChildren() throws HiLAException {
					List<Resource> children = new ArrayList<Resource>();
					return children;
				}

				/**
				 * @see eu.unicore.hila.Resource#ok()
				 */
				@Override
				public boolean ok() {
					return true;
				}
			};
		} catch (HiLALocationSyntaxException e) {
			new Fail(e);
		}
	}

	/**
	 * Test method for
	 * {@link eu.unicore.hila.common.BaseResource#BaseResource(eu.unicore.hila.Location)}
	 * .
	 */
	@Test
	public void testBaseResource() {

		List<Resource> children;
		try {
			children = res.getChildren();
			Assert.assertEquals(0, children.size());
		} catch (HiLAException e) {
			new Fail(e);
		}
	}

	/**
	 * Test method for {@link eu.unicore.hila.common.BaseResource#getLocation()}
	 * .
	 */
	@Test
	public void testGetLocation() {
		Location loc = res.getLocation();
		try {
			Assert.assertEquals(new Location(RES_LOCATION), loc);
		} catch (HiLALocationSyntaxException e) {
			new Fail(e);
		}
	}

	/**
	 * Test method for
	 * {@link eu.unicore.hila.common.BaseResource#getChild(java.lang.String)}.
	 * 
	 * I don't think this can be tested.
	 */
	public void testGetChild() {
		try {
			Resource child = res.getChild("child");
			Assert.assertEquals(res.getLocation().getChildLocation("child"),
					child.getLocation());
		} catch (HiLAException e) {
			new Fail(e);
		}
	}

	/**
	 * Test method for {@link eu.unicore.hila.common.BaseResource#getMetadata()}
	 * .
	 */
	@Test
	public void testGetMetadata() {
		try {
			Metadata md = res.getMetadata();
			Assert.assertNotNull(md);
		} catch (HiLAException e) {
			new Fail(e);
		}
	}

	/**
	 * Test method for {@link eu.unicore.hila.common.BaseResource#getName()}.
	 */
	@Test
	public void testGetName() {
		String name = res.getName();
		Assert.assertEquals(res.getLocation().getName(), name);
	}

	/**
	 * Test method for {@link eu.unicore.hila.common.BaseResource#getParent()}.
	 */
	@Test
	public void testGetParent() {
		try {
			Resource parent = res.getParent();
			Location parentLoc = res.getLocation().getParentLocation();
			Assert.assertEquals(parentLoc.locate(), parent);
		} catch (HiLAException e) {
			new Fail(e);
		}
	}

	/**
	 * Test method for {@link BaseResource#ok()}
	 */
	@Test
	public void testOk() {
		Assert.assertTrue(res.ok());
	}

	/**
	 * Test method for
	 * {@link eu.unicore.hila.common.BaseResource#findParentLocationOfType(java.lang.Class, eu.unicore.hila.Location, java.lang.Class)}
	 * .
	 * 
	 * Cannot be tested. No types defined.
	 */
	// @Test
	public void testFindParentLocationOfType() {

	}

}
