/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.Set;

import org.apache.log4j.Logger;

import eu.unicore.hila.annotations.ResourceType;

/**
 * 
 * @author bjoernh
 * 
 */
public class ResourceTypeRegistry {

	private static volatile ResourceTypeRegistry theInstance = null;
	private static final Logger log = Logger
			.getLogger(ResourceTypeRegistry.class);

	public static ResourceTypeRegistry getInstance() {
		if (theInstance == null) {
			synchronized (ResourceTypeRegistry.class) {
				if (theInstance == null) {
					theInstance = new ResourceTypeRegistry();
				}
			}
		}
		return theInstance;
	}

	private Map<LocationPattern, Class<Resource>> locationTypeMap;

	public Class<Resource> determineResourceType(Location location) {
		LocationPattern lp = determineLocationPattern(location);
		if (lp != null) {
			return this.locationTypeMap.get(lp);
		}

		// TODO think of throwing an exception for "unknown resource type"
		return null;
	}

	public LocationPattern determineLocationPattern(Location location) {
		for (LocationPattern lp : this.locationTypeMap.keySet()) {
			if (lp.match(location)) {
				return lp;
			}
		}
		// TODO think of throwing an exception
		return null;
	}

	private ResourceTypeRegistry() {
		initialize();
	}

	/**
     * 
     */
	private void initialize() {
		List<Class<Resource>> resourceTypes = new ArrayList<Class<Resource>>();

		ServiceLoader<ImplementationProvider> sl = ServiceLoader
				.load(ImplementationProvider.class);
		Iterator<ImplementationProvider> iter = sl.iterator();
		while (iter.hasNext()) {
			List<Package> packages = iter.next().getPackages();
			resourceTypes.addAll(findResourceTypes(packages));
		}
		this.locationTypeMap = buildLocationTypeMap(resourceTypes);
	}

	/**
	 * @param resourceTypes
	 * @return
	 */
	private Map<LocationPattern, Class<Resource>> buildLocationTypeMap(
			List<Class<Resource>> resourceTypes) {
      Map<LocationPattern, Class<Resource>> locationTypeMap = new LinkedHashMap<LocationPattern, Class<Resource>>(
				resourceTypes.size());

		for (Class<Resource> class1 : resourceTypes) {
			String[] stringLocationPatterns = class1.getAnnotation(
					ResourceType.class).locationStructure();
			for (String string : stringLocationPatterns) {
				LocationPattern lp = new LocationPattern(string);
				locationTypeMap.put(lp, class1);
			}
		}

		return locationTypeMap;
	}

	/**
	 * @param packages
	 * @return
	 */
	private List<Class<Resource>> findResourceTypes(List<Package> packages) {
		List<Class<Resource>> resourceTypes = new ArrayList<Class<Resource>>();
		for (Package package1 : packages) {
			resourceTypes.addAll(findResourceTypes(package1));
		}
		return resourceTypes;
	}

	private List<Class<Resource>> findResourceTypes(Package package1) {
		// return findResourceTypes1(package1);
		// return findResourceTypes2(package1);
		return findResourceTypes3(package1);
	}

	/**
	 * @param package1
	 * @return
	 */
	private List<Class<Resource>> findResourceTypes3(Package package1) {
		List<Class<Resource>> resourceTypes = new ArrayList<Class<Resource>>();

		try {
			Set<Class<?>> classes = PackageUtil.getClasses(package1.getName());
			for (Class<?> class1 : classes) {
				if (class1.isAnnotationPresent(ResourceType.class)) {
					resourceTypes.add((Class<Resource>) class1);
				}
			}
		} catch (Exception e) {
			log.error("Unable to get set of classes from package.", e);
		}

		return resourceTypes;
	}
}
