/*********************************************************************************
 * Copyright (c) 2009 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.client.StorageClient;
import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.grid.BaseStorage;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.grid.Site;
import eu.unicore.hila.grid.Storage;

/**
 * @author bjoernh
 * 
 *         25.08.2009 15:44:19
 * 
 */
@ResourceType(locationStructure = {
	"unicore6:/sites/{site}/storages/{storage}/?",
	"unicore6:/{user}@sites/{site}/storages/{storage}/?",
	"unicore6:/sites/{site}/tasks/{task}/wd/?",
	"unicore6:/{user}@sites/{site}/tasks/{task}/wd/?",
	"unicore6:/storages/{storage}/?",
	"unicore6:/{user}@storages/{storage}/?" })
public class Unicore6Storage extends BaseStorage implements Storage {

    private static final Logger log = Logger.getLogger(Unicore6Storage.class);

   // TODO replace with EHCache
    private static final Map<Location, Unicore6Storage> storages = new HashMap<Location, Unicore6Storage>();

    private StorageClient storageClient = null;

    private EndpointReferenceType storageEPR;

    private Unicore6SecurityProperties sec;

    private Unicore6Storage(Location _location, Object... _extraInformation)
	    throws HiLAException {
	super(_location);
	for (Object object : _extraInformation) {
	    if (object instanceof EndpointReferenceType) {
		this.storageEPR = (EndpointReferenceType) object;
	    } else if (object instanceof Unicore6SecurityProperties) {
		this.sec = (Unicore6SecurityProperties) object;
	    } else if (object instanceof StorageClient) {
		this.storageClient = (StorageClient) object;
	    }
	}
	if (storageClient != null) {
	    return;
	} else if ((storageEPR != null) && (sec != null)) {
	    try {
		storageClient = new StorageClient(storageEPR, sec);
	    } catch (Exception e) {
		throw new HiLAException(
			"Unable to instantiate client for storage: "
				+ this.location, e);
	    }
	} else {
	    throw new HiLAException(
		    "Insufficient information to create storage client.");
	}
    }

    public static synchronized Unicore6Storage locate(Location _location,
	    Object... _extraInformation) throws HiLAException {
	if (storages.containsKey(_location)) {
	    return storages.get(_location);
	} else {
	    boolean containsEPR = false;
	    boolean containsSecProp = false;
	    boolean containsStorageClient = false;
	    if (_extraInformation != null) {
		for (Object object : _extraInformation) {
		    if (object instanceof EndpointReferenceType) {
			containsEPR = true;
		    }
		    if (object instanceof Unicore6SecurityProperties) {
			containsSecProp = true;
		    }
		    if (object instanceof StorageClient) {
			containsStorageClient = true;
		    }
		}
	    }
	    if (!(containsEPR && containsSecProp) && !containsStorageClient) {
		try {
		    final Location taskLocation = findParentLocationOfType(
			    Unicore6Job.class, _location,
			    Unicore6Grid.class);
		    Unicore6Job task = (Unicore6Job) taskLocation
			    .locate();
		    if (task != null) {
			StorageClient storageClient = task.jobClient
				.getUspaceClient();
			Unicore6Storage storage = new Unicore6Storage(
				_location, storageClient);
			storages.put(_location, storage);
			return storage;
		    }
		} catch (HiLAException e) {
		    // this storage is not a working directory
		} catch (Exception e) {
		    // getUspaceClient failed
		}
		final Location siteLocation = findParentLocationOfType(
			Unicore6Site.class, _location, Unicore6Grid.class);
		return (Unicore6Storage) ((Site) siteLocation.locate())
			.getStorage(_location.getName());
	    }
	    final Unicore6Storage storage = new Unicore6Storage(_location,
		    _extraInformation);
	    storages.put(_location, storage);
	    return storage;
	}
    }

    /**
     * @return
     */
    protected StorageClient getStorageClient() {
	return storageClient;
    }

    /**
     * @see eu.unicore.hila.common.grid.BaseStorage#getChildren()
     */
    @Override
    public List<Resource> getChildren() throws HiLAException {
	List<Resource> children = new ArrayList<Resource>();

	children.addAll(super.getChildren());

	try {
	    children.add(this.location.getChildLocation("exports").locate());
	} catch (Exception e) {
	    log.error(e.getMessage(), e);
	}
	try {
	    children.add(this.location.getChildLocation("imports").locate());
	} catch (Exception e) {
	    log.error(e.getMessage(), e);
	}
	try {
	    children.add(this.location.getChildLocation("transfers").locate());
	} catch (Exception e) {
	    log.error(e.getMessage(), e);
	}

	return children;
    }
}
