/*********************************************************************************
 * Copyright (c) 2009-2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.apache.log4j.Logger;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.TargetSystemFactory;
import eu.unicore.hila.HiLA;
import eu.unicore.hila.Location;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.BaseResource;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.grid.Config;
import eu.unicore.hila.grid.Site;

/**
 * @author bjoernh
 * 
 *         13.08.2009 15:25:37
 * 
 */
@ResourceType(locationStructure = { "unicore6:/sites/?",
      "unicore6:/{user}@sites/?" })
public class Unicore6SitesCollection extends BaseResource implements Resource {
   private static final Logger log = Logger
         .getLogger(Unicore6SitesCollection.class);

   private static final Map<Location, Unicore6SitesCollection> sitesCollections = new HashMap<Location, Unicore6SitesCollection>();

   private static ScheduledExecutorService executor = HiLA
         .getDaemonExecutor();

   private List<Resource> sites;

   private Config config = null;
   private Object configLock = new Object();

   private SitesCollectionUpdater updater;

   private boolean initialized = false;

   private HiLAException lastError;

   /**
    * @param location
    * @throws HiLAException
    */
   private Unicore6SitesCollection(Location location) throws HiLAException {
      super(location);

      sites = new ArrayList<Resource>();

      updater = new SitesCollectionUpdater(this);

      String sUpdateInterval = ((Unicore6Grid) getParent()).getProperties()
            .getProperty(Unicore6Properties.SITE_UPDATE_INTERVAL, "60000");
      long updateInterval = Long.parseLong(sUpdateInterval);
      executor.scheduleAtFixedRate(updater, 0, updateInterval,
            TimeUnit.MILLISECONDS);
   }

   public static Unicore6SitesCollection locate(Location _location,
         Object... _extraInformation) throws HiLAException {
      if (sitesCollections.containsKey(_location)) {
         return sitesCollections.get(_location);
      }
      Unicore6SitesCollection sitesCollection = new Unicore6SitesCollection(
            _location);
      sitesCollections.put(_location, sitesCollection);
      return sitesCollection;
   }

   /**
    * @throws HiLAException
    * @see eu.unicore.hila.Resource#getChildren()
    */
   public synchronized List<Resource> getChildren() throws HiLAException {
      if(!initialized && lastError!=null){
	  throw new HiLAException("An error occured during site update",lastError);
      }
      if (!initialized) {
         try {
            wait();
         } catch (InterruptedException e) {
         }
      }
      return sites;

      // final Unicore6Grid parent = (Unicore6Grid) getParent();
      // final Unicore6Properties props = parent.getProperties();

      // List<String> registryURLs = props.getRegistryUrls();

      // if (sites.isEmpty()
      // || (System.currentTimeMillis() - sitesTimestamp) > Long
      // .parseLong(props.getProperty("hila.unicore6.refresh",
      // "60000"))) {
      // if (log.isDebugEnabled()) {
      // log.debug("Updating sites list.");
      // }
      // sites.clear();
      // sites.addAll(getAllSites(registryURLs, props, location));
      //
      // sitesTimestamp = System.currentTimeMillis();
      // }

      // return sites;
   }

   protected synchronized void setSites(List<Site> _sites) {
      sites.clear();
      sites.addAll(_sites);
      initialized = true;
      clearError();
      notify();
   }

   protected synchronized void setError(HiLAException _exception) {
      this.lastError = _exception;
   }

   private synchronized void clearError() {
      this.lastError = null;
   }

   private class SitesCollectionUpdater implements Runnable {

      private Unicore6SitesCollection sitesCollection;
      private List<String> registryURLs;
      private Unicore6Properties props;

      /**
       * @param _sitesCollection
       * @throws HiLAException
       * 
       */
      public SitesCollectionUpdater(Unicore6SitesCollection _sitesCollection)
            throws HiLAException {
         this.sitesCollection = _sitesCollection;
         Unicore6Grid grid = (Unicore6Grid) sitesCollection.getParent();
         this.props = grid.getProperties();
         this.registryURLs = this.props.getRegistryUrls();

      }

      /**
       * @see java.lang.Runnable#run()
       */
      @Override
      public void run() {
         /*
          * timeout can actually become twice as long, because it is used twice.
          * Minimum timeouts for LAN connections are 2500L if a TargetSystem is
          * already existing and 4000L if a TargetSystem needs to be created for
          * remote connections
          */
         final long timeoutMillis = Long.parseLong(props.getProperty(
               "hila.unicore6.timeout", "5000"));

         if (config == null) {
            synchronized (configLock) {
               if (config == null) {
                  String configProviderClass = props.getConfigImplementation();
                  Class<Config> configClass;
                  try {
                     configClass = (Class<Config>) Class
                           .forName(configProviderClass);
                  } catch (ClassNotFoundException e) {
                     setError(new HiLAException(
                           "hila.unicore6.config references unknown class"
                                 + configProviderClass, e));
                     return;
                  } catch (ClassCastException e) {
                     setError(new HiLAException(configProviderClass
                           + " is not a subclass of " + Config.class.getName()));
                     return;
                  }
                  Constructor<Config> constructor;
                  try {
                     constructor = configClass
                           .getConstructor(Unicore6Properties.class);
                  } catch (SecurityException e) {
                     setError(new HiLAException(e.getMessage(), e));
                     return;
                  } catch (NoSuchMethodException e) {
                     setError(new HiLAException(
                           "Config implementation referenced in "
                                 + "hila.unicore6.config does not provide the "
                                 + "correct constructor", e));
                     return;
                  }
                  try {
                     config = constructor.newInstance(props);
                  } catch (IllegalArgumentException e) {
                     setError(new HiLAException(e.getMessage(), e));
                     return;
                  } catch (InstantiationException e) {
                     setError(new HiLAException(e.getMessage(), e));
                     return;
                  } catch (IllegalAccessException e) {
                     setError(new HiLAException(e.getMessage(), e));
                     return;
                  } catch (InvocationTargetException e) {
                     setError(new HiLAException(e.getMessage(), e));
                     return;
                  }
               }
            }

         }

         Unicore6SecurityProperties secProps;
         try {
            secProps = (Unicore6SecurityProperties) config
                  .getIdForLocation(location);
         } catch (HiLAException e1) {
            setError(e1);
            return;
         }

         if (secProps != null) {
            secProps = (Unicore6SecurityProperties) secProps.clone();
         }

         List<Site> sites = new ArrayList<Site>();
         // ExecutorService executor = Executors.newCachedThreadPool();
         List<Future<List<EndpointReferenceType>>> tsfFutures = new ArrayList<Future<List<EndpointReferenceType>>>();
         for (String registryURL : registryURLs) {
            GetServicesFromRegistry getTSFs = new GetServicesFromRegistry(
                  registryURL, secProps, TargetSystemFactory.TSF_PORT);
            tsfFutures.add(executor.submit(getTSFs));
         }

         final long tsfDeadline = System.currentTimeMillis() + timeoutMillis;
         // take results and timeout
         for (Future<List<EndpointReferenceType>> tsfFuture : tsfFutures) {
            try {
               List<Future<List<Site>>> siteFutures = new ArrayList<Future<List<Site>>>();
               List<EndpointReferenceType> tsfEprs = tsfFuture.get(tsfDeadline
                     - System.currentTimeMillis(), TimeUnit.MILLISECONDS);
               for (EndpointReferenceType endpointReferenceType : tsfEprs) {
                  GetSitesFromTSF getSites = new GetSitesFromTSF(
                        endpointReferenceType, secProps, location,
                        props.getSeparateTSSs());
                  siteFutures.add(executor.submit(getSites));
               }

               final long siteDeadline = System.currentTimeMillis()
                     + timeoutMillis;
               for (Future<List<Site>> siteFuture : siteFutures) {
                  try {
                     List<Site> tsfSites = siteFuture.get(
                           siteDeadline - System.currentTimeMillis(),
                           TimeUnit.MILLISECONDS);
                     sites.addAll(tsfSites);
                  } catch (InterruptedException e) {
                     log.warn("Interrupted while waiting for list of sites.", e);
                  } catch (ExecutionException e) {
                     log.warn("Error retrieving target systems.", e);
                  } catch (TimeoutException e) {
                     log.warn("Retrieval of target systems timed out.", e);
                  }
               }

            } catch (InterruptedException e) {
               log.warn(
                     "Interrupted while waiting for list of TargetSystemFactory.",
                     e);
            } catch (ExecutionException e) {
               log.warn("Error retrieving target system factories.", e);
            } catch (TimeoutException e) {
               log.warn("Retrieval of target system factories timed out.", e);
            }
         }

         setSites(sites);
      }
   }
}
