/*********************************************************************************
 * Copyright (c) 2011 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
package eu.unicore.hila.grid.unicore6;

import java.util.Collections;
import java.util.List;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;

import de.fzj.unicore.uas.client.ReservationClient;

import de.fzj.unicore.wsrflite.xfire.ClientException;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceNotDestroyedFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnavailableFault;
import de.fzj.unicore.wsrflite.xmlbeans.exceptions.ResourceUnknownFault;
import eu.unicore.hila.Location;
import eu.unicore.hila.Metadata;
import eu.unicore.hila.Resource;
import eu.unicore.hila.annotations.ResourceType;
import eu.unicore.hila.common.BaseResource;
import eu.unicore.hila.exceptions.HiLAException;
import eu.unicore.hila.exceptions.HiLAFactoryException;
import eu.unicore.hila.exceptions.HiLALocationPatternException;
import eu.unicore.hila.grid.Grid;
import eu.unicore.hila.grid.Reservable;
import eu.unicore.hila.grid.Reservation;
import eu.unicore.hila.grid.Site;

/**
 * @author bjoernh
 * @since 2.2
 *         17.01.2011 14:39:43
 * 
 */
@ResourceType(locationStructure = {
	"unicore6:/sites/{siteName}/reservations/{reservation}/?",
	"unicore6:/{user}@sites/{siteName}/reservations/{reservation}/?" })
public class Unicore6Reservation extends BaseResource implements Reservation {

    private ReservationClient resClient;

    private Metadata metadata;

    private static Cache reservationsCache;
    static {
	final CacheManager manager = CacheManager.getInstance();
	manager.addCache(new Cache(Unicore6Reservation.class.getName(), 100,
		false, false, 300, 60));
	reservationsCache = manager.getCache(Unicore6Reservation.class
		.getName());
    }

    /**
     * @param _location
     * @param resClient
     */
    private Unicore6Reservation(Location _location, ReservationClient _resClient) {
	super(_location);
	this.metadata = new Metadata();
	this.resClient = _resClient;
    }

    public static synchronized Resource locate(Location _location,
	    Object... _extraInformation) throws HiLAException {
	if (reservationsCache.isKeyInCache(_location)) {
	    final Element cacheElem = reservationsCache.get(_location);
	    if (cacheElem != null) {
		return (Resource) cacheElem.getObjectValue();
	    }
	}
	ReservationClient reservationClient = null;
	for (Object object : _extraInformation) {
	    if (object instanceof ReservationClient) {
		reservationClient = (ReservationClient) object;
	    }
	}
	if (reservationClient == null) {
	    // throw new HiLAFactoryException(
	    // "Cannot instantiate reservation resource.");
	    try {
		Reservation reservation = ((Reservable) findParentLocationOfType(
			Site.class, _location, Grid.class).locate())
			.getReservation(_location.getName());
		if (reservation != null) {
		    return reservation;
		} else {
		    throw new HiLAFactoryException("Reservation " + _location
			    + " does not exist.");
		}
	    } catch (HiLALocationPatternException e) {
		throw new HiLAFactoryException(
			Unicore6Reservation.class.getName()
				+ "#locate() created wrong location.");
	    }
	}
	Unicore6Reservation u6res = new Unicore6Reservation(_location,
		reservationClient);
	reservationsCache.put(new Element(_location, u6res));
	return u6res;
    }

    /**
     * @see eu.unicore.hila.Resource#getChildren()
     */
    @Override
    public List<Resource> getChildren() throws HiLAException {
	return Collections.emptyList();
    }

    /**
     * @see eu.unicore.hila.grid.Reservation#cancel()
     */
    @Override
    public void cancel() throws HiLAException {
	try {
	    resClient.destroy();
	} catch (BaseFault e) {
	    throw new HiLAException("Unable to cancel reservation.", e);
	} catch (ResourceUnavailableFault e) {
	    throw new HiLAException("Reservation does not exist.", e);
	} catch (ResourceUnknownFault e) {
	    throw new HiLAException("Reservation does not exist.", e);
	} catch (ResourceNotDestroyedFault e) {
	    throw new HiLAException("Unable to cancel reservation.", e);
	} catch (ClientException e) {
	    throw new HiLAException("Client exception.", e);
	}

    }

    /**
     * @see eu.unicore.hila.common.BaseResource#getMetadata()
     */
    @Override
    public Metadata getMetadata() throws HiLAException {
	updateMetadata();

	return this.metadata;
    }

    /**
     * @throws HiLAException
     * 
     */
    private void updateMetadata() throws HiLAException {
	try {
	    this.metadata.setData("resourceProperties",
		    resClient.getResourcePropertyDocument());
	} catch (BaseFault e) {
	    throw new HiLAException("Unable to retrieve resourceProperties.", e);
	} catch (ResourceUnknownFault e) {
	    throw new HiLAException("Unknown resource.", e);
	} catch (ResourceUnavailableFault e) {
	    throw new HiLAException("Unavailable resource.", e);
	} catch (ClientException e) {
	    throw new HiLAException("Client exception.", e);
	}
    }

    /**
     * @see eu.unicore.hila.grid.Reservation#getId()
     */
    @Override
    public String getId() throws HiLAException {
	try {
	    return resClient.getReservationReference();
	} catch (Exception e) {
	    throw new HiLAException("Cannot determine reservation id.", e);
	}
    }
}
